package org.exploit.blockbook;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.module.SimpleModule;
import org.eclipse.jetty.http.HttpMethod;
import org.exploit.blockbook.deserializer.BlockBookTransactionDeserializer;
import org.exploit.blockbook.deserializer.PushResultDeserializer;
import org.exploit.blockbook.model.*;
import org.exploit.blockbook.model.address.BlockBookAddress;
import org.exploit.blockbook.model.address.XPubAddress;
import org.exploit.blockbook.model.error.BlockBookErrorMessage;
import org.exploit.blockbook.model.result.PushResult;
import org.exploit.blockbook.model.transaction.BlockBookTransaction;
import org.exploit.finja.core.model.WebData;
import org.exploit.finja.exception.FinjaException;
import org.exploit.finja.utils.Jackson;
import org.exploit.jettyx.annotation.Body;
import org.exploit.jettyx.annotation.HttpRequest;
import org.exploit.jettyx.annotation.Path;
import org.exploit.jettyx.annotation.Query;
import org.exploit.jettyx.exception.InvalidResponseBodyException;
import org.exploit.jettyx.model.HttpResponse;

import java.io.IOException;
import java.util.List;

public interface BlockBookApi {
    @HttpRequest(path = "/api")
    HttpResponse<Status> getStatus();

    @HttpRequest(path = "/api/status")
    HttpResponse<Status> getApiStatus();

    @HttpRequest(path = "/api/v2/block-index/{height}")
    HttpResponse<BlockHash> getBlockHash(@Path("height") int height);

    @HttpRequest(path = "/api/v2/tx/{txid}")
    HttpResponse<BlockBookTransaction> getTransaction(@Path("txid") String txid);

    @HttpRequest(path = "/api/v2/tx/{txid}")
    HttpResponse<BlockBookTransaction> getTransactionR(@Path("txid") String txid);

    @HttpRequest(path = "/api/v2/address/{address}")
    HttpResponse<BlockBookAddress> getAddress(@Path("address") String address, @Query("page") int page);

    @HttpRequest(path = "/api/v2/xpub/{xpub}")
    HttpResponse<XPubAddress> getXPub(@Path("xpub") String xpub, @Query("page") int page);

    @HttpRequest(path = "/api/v2/block/{block}")
    HttpResponse<Block> getBlock(@Path("block") int block, @Query("page") int page);

    @HttpRequest(path = "/api/v2/utxo/{address}")
    HttpResponse<List<UnspentOutput>> getUnspentOutputs(@Path("address") String address);

    @HttpRequest(path = "/api/v2/sendtx/", method = HttpMethod.POST)
    HttpResponse<JsonNode> push(@Body String hex);

    @HttpRequest(path = "/api/v2/balancehistory/{address}")
    HttpResponse<List<BalanceHistoryEntry>> getBalanceHistory(@Path("address") String address);

    static BlockBookApi create(WebData webData) {
        var blockBookModule = new SimpleModule()
                .addDeserializer(BlockBookTransaction.class, new BlockBookTransactionDeserializer())
                .addDeserializer(PushResult.class, new PushResultDeserializer());

        Jackson.getObjectMapper().registerModule(blockBookModule);

        return webData.newApiClient(BlockBookApi.class);
    }

    static <T> T handle(HttpResponse<T> response) {
        if (response.exception() != null && response.exception() instanceof InvalidResponseBodyException resp) {
            try {
                var errorMessage = BlockBookErrorMessage.from(response.rawResponse().getContent());
                throw new FinjaException(errorMessage.error());
            } catch (IOException e) {
                throw resp;
            }
        }

        return response.body();
    }
}