package org.exploit.blockbook;

import lombok.RequiredArgsConstructor;
import org.exploit.blockbook.model.address.BlockBookAddress;
import org.exploit.finja.core.EventFetcher;
import org.exploit.finja.core.event.TxnEvent;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;

import java.time.Instant;
import java.util.ArrayList;
import java.util.List;

@RequiredArgsConstructor
public class BlockBookEventFetcher implements EventFetcher {
    private final BlockBookApi api;
    private final BlockBookEventTransformer eventTransformer;

    @Override
    public Flux<TxnEvent> events(String address, long startTimestamp) {
        var timestampSeconds = Instant.ofEpochMilli(startTimestamp).getEpochSecond();

        return Mono.fromCallable(() -> BlockBookApi.handle(api.getAddress(address, 1)))
                .flatMap(this::fetchTxids)
                .flatMapMany(transactions -> process(address, timestampSeconds, transactions));
    }

    private Flux<TxnEvent> process(String address, long start, List<String> transactions) {
        return Flux.fromIterable(transactions)
                .flatMapSequential(txid -> Mono.fromCallable(() -> BlockBookApi.handle(api.getTransaction(txid))))
                .takeWhile(tx -> tx.getBlockTime() >= start)
                .flatMapIterable(tx -> eventTransformer.transform(address, tx));
    }

    private Mono<List<String>> fetchTxids(BlockBookAddress info) {
        var address = info.getAddress();
        var totalPages = info.getTotalPages();
        var txids = info.getTxids();

        if (txids == null) {
            return Mono.just(new ArrayList<>());
        }

        var transactions = new ArrayList<>(txids);

        if (totalPages > 1) {
            return Flux.range(2, totalPages - 1)
                    .flatMap(i -> Mono.fromCallable(() -> BlockBookApi.handle(api.getAddress(address, i))))
                    .doOnNext(addr -> transactions.addAll(addr.getTxids()))
                    .then()
                    .thenReturn(transactions);
        }

        return Mono.just(transactions);
    }
}