package beaver.comp.run;

import org.extendj.Trace;
import org.extendj.neobeaver.BeaverParser;
import org.extendj.neobeaver.Grammar;
import org.extendj.neobeaver.MyParser;
import org.extendj.neobeaver.Parser;
import org.extendj.neobeaver.Scanner;

import java.io.File;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.PrintStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;

public class Make {
  public static void main(String[] args) throws IOException {
    System.out.println("Welcome to NeoBeaver classic frontend!");

    String destDirName = "";
    String filename = "";

    for (int i = 0; i < args.length; ++i) {
      String arg = args[i];
      switch (arg) {
        case "-h":
          System.out.println("Usage: nbfront [OPTIONS] <PARSER SPEC FILE>");
          System.out.println("Options:");
          System.out.println("  -d <DIR>    Generate parser in the named directory, DIR.");
          System.out.println("  -h          Print help and exit.");
          System.exit(0);
          return;
        case "-d":
          if (args.length > i + 1) {
            destDirName = args[i + 1];
            i += 1;
          } else {
            System.err.println("Error: missing path argument to -d option.");
          }
          break;
        case "-t":
          // Generate terminal names.
          // TODO: implement (always on right now).
          break;
        case "-c":
          // Suppress parsing table compression.
          // TODO: implement.
          break;
        case "-w":
          // Use switch to invoke reduce actions.
          // TODO: implement.
          break;
        default:
          if (arg.startsWith("-")) {
            System.err.format(
                "Warning: ignoring option '%s'. It is probably an unsupported class Beaver flag.%n",
                arg);
          } else {
            if (filename.isEmpty()) {
              filename = arg;
            } else {
              System.err.format("Warning: multiple filenames specified, ignoring file %s%n", arg);
            }
          }
      }
    }

    if (filename.isEmpty()) {
      System.err.println("Error: no parser specification filename specified!");
      System.exit(1);
      return;
    }

    Trace trace = new Trace("NeoBeaverFrontend");
    try (FileReader reader = new FileReader(new File(filename));
        Scanner scanner = new Scanner(reader)) {
      Parser.Symbol parseTree;
      try {
        trace.pushEvent("parseGrammar");
        Parser grammarParser = new Parser();
        parseTree = grammarParser.parse(scanner);
      } finally {
        trace.popEvent();
      }
      Grammar grammar;
      try {
        trace.pushEvent("BP.buildParser");
        grammar = BeaverParser.buildParser(trace, parseTree);
      } finally {
        trace.popEvent();
      }
      try {
        trace.pushEvent("checkProblems");
        grammar.checkProblems();
      } finally {
        trace.popEvent();
      }
      MyParser parser;
      try {
        trace.pushEvent("Grammar.buildParser");
        parser = grammar.buildParser(trace);
      } finally {
        trace.popEvent();
      }
      Path destPath;
      if (destDirName.isEmpty()) {
        destPath = Paths.get("");
      } else {
        destPath = Paths.get(destDirName);
        if (!Files.exists(destPath)) {
          System.err.format("Error: the named destination directory %s does not exist.%n",
              destDirName);
          System.exit(1);
          return;
        }
      }
      if (!grammar.packageName.isEmpty()) {
        destPath = destPath.resolve(grammar.packageName.replaceAll("\\.", File.separator));
      }
      if (!Files.exists(destPath)) {
        System.out.println("Creating output directory: " + destPath);
        Files.createDirectories(destPath);
      }
      File dest = destPath.resolve(String.format("%s.java", grammar.className)).toFile();
      try (FileOutputStream fout = new FileOutputStream(dest);
          PrintStream out = new PrintStream(fout)) {
        grammar.printBeaverParser(out, parser);
      }
      parser.transitions.printConflicts(grammar);
    } catch (BeaverParser.SpecError e) {
      // TODO: improve error handling (multiple errors).
      System.err.println("Error in parser specification:");
      System.err.println(e.getMessage());
      System.exit(1);
    }
  }
}
