/* Copyright (c) 2017, Jesper Öqvist
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. Neither the name of the copyright holder nor the names of its
 * contributors may be used to endorse or promote products derived from this
 * software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */
package org.extendj.neobeaver;

import org.extendj.Trace;

import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.util.Collection;
import java.util.HashSet;
import java.util.Set;

public class NeoBeaver {
  public static void main(String[] args) throws IOException {
    Set<String> files = new HashSet<>();
    Set<String> options = new HashSet<>();
    for (String arg : args) {
      if (arg.startsWith("-")) {
        options.add(arg);
      } else {
        files.add(arg);
      }
    }

    if (args.length < 1) {
      System.err.println("No input files specified!");
      System.exit(1);
      return;
    }

    try {
      run(files, options);
    } catch (BeaverParser.SpecError e) {
      // TODO: improve error handling (multiple errors).
      System.err.println("Error in parser specification:");
      System.err.println(e.getMessage());
      System.exit(1);
    }
  }

  public static void run(Collection<String> files, Set<String> options) throws IOException {
    Trace trace = new Trace("NeoBeaver");
    try {
      if (options.contains("--tokenize") || options.contains("-t")) {
        for (String filename : files) {
          try (Scanner scanner = new Scanner(new FileReader(new File(filename)))) {
            while (true) {
              Parser.Token sym = scanner.nextToken();
              System.out.format("%4d: %s%n", sym.getId(), sym);
              if (sym.getId() == Parser.Tokens.EOF) {
                break;
              }
            }
          }
        }
      } else {
        for (String filename : files) {
          try (Scanner scanner = new Scanner(new FileReader(new File(filename)))) {
            Parser.Symbol parseTree;
            try {
              trace.pushEvent("parseGrammar");
              Parser grammarParser = new Parser();
              parseTree = grammarParser.parse(scanner);
            } finally {
              trace.popEvent();
            }
            Grammar grammar;
            try {
              trace.pushEvent("BP.buildParser");
              grammar = BeaverParser.buildParser(trace, parseTree);
            } finally {
              trace.popEvent();
            }
            try {
              trace.pushEvent("checkProblems");
              grammar.checkProblems();
            } finally {
              trace.popEvent();
            }
            MyParser parser;
            try {
              trace.pushEvent("Grammar.buildParser");
              parser = grammar.buildParser(trace);
            } finally {
              trace.popEvent();
            }
            if (options.contains("--print-table") || options.contains("-table")) {
              parser.transitions.printTables(parser.grammar, parser.itemSets);
              parser.printTables();
            }
            if (options.contains("--print-table") || options.contains("-p")) {
              parser.printTables();
            }
            if (options.contains("-i")) {
              parser.printItems();
            }
            if (options.contains("-g")) {
              parser.printGraph();
            }
            if (options.contains("--beaver")) {
              parser.printBeaverParser();
            }
            if (options.contains("--standalone") || options.contains("-s")) {
              // Generate standalone parser.
              parser.printParser();
            }
            if (options.contains("--classic")) {
              // Print classic Beaver specification.
              parser.printBeaverSpec();
            }
            if (options.contains("--test")) {
              // Print classic Beaver specification.
              parser.printBeaverTestSpec();
            }
            parser.transitions.printConflicts(grammar);
          }
        }
      }
    } finally {
      if (options.contains("--trace")) {
        trace.sendTo("localhost", 8040);
      }
    }
  }

}
