/* Copyright (c) 2017, Jesper Öqvist
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. Neither the name of the copyright holder nor the names of its
 * contributors may be used to endorse or promote products derived from this
 * software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */
package org.extendj.neobeaver;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

public class TransitionTable {
  public final ItemSet goal;
  public final Map<ItemSet, Map<Symbol, ItemSet>> map;
  public final Map<ItemSet, Map<Symbol, Action>> actions;
  public final List<Conflict> conflicts;

  public TransitionTable(ItemSet goal, Map<ItemSet, Map<Symbol, ItemSet>> map,
      Map<ItemSet, Map<Symbol, Action>> actions, List<Conflict> conflicts) {
    this.goal = goal;
    this.map = map;
    this.actions = actions;
    this.conflicts = conflicts;
  }

  public static TransitionTable build(Grammar grammar, Set<Symbol> syms, List<ItemSet> itemSets,
      ItemSet goal, Map<ItemSet, ItemSet> coreMap,
      List<Tuple3<ItemSet, ItemSet, Symbol>> transitions,
      List<Tuple3<ItemSet, Symbol, Action>> extraActions) {
    Map<ItemSet, Map<Symbol, ItemSet>> map = new HashMap<>();
    Map<ItemSet, Map<Symbol, Action>> actions = new HashMap<>();
    for (ItemSet set : itemSets) {
      map.put(set, new HashMap<Symbol, ItemSet>());
      actions.put(set, new HashMap<Symbol, Action>());
    }
    for (Tuple3<ItemSet, ItemSet, Symbol> transition : transitions) {
      ItemSet source = coreMap.get(transition.first);
      ItemSet dest = coreMap.get(transition.second);
      Symbol sym = transition.third;
      map.get(source).put(sym, dest);
    }
    List<Symbol> nonterminals = new ArrayList<>();
    List<Symbol> terminals = new ArrayList<>();
    List<Conflict> conflicts = new ArrayList<>();
    for (Symbol sym : syms) {
      if (sym.isTerminal()) {
        terminals.add(sym);
      } else {
        nonterminals.add(sym);
      }
    }
    for (ItemSet set : itemSets) {
      for (Item item : set.allItems()) {
        if (item.dot >= item.rule.rhs.size()) {
          Collection<Tuple3<ItemSet, Symbol, Action>> reduceActs = item.reduceActions(grammar, set);
          for (Tuple3<ItemSet, Symbol, Action> action : reduceActs) {
            addAction(grammar, actions, conflicts, action.first, action.second, action.third);
          }
        }
      }
      Map<Symbol, ItemSet> setTransitions = map.get(set);
      for (Symbol sym : terminals) {
        if (setTransitions.containsKey(sym)) {
          addAction(grammar, actions, conflicts, set, sym, new Shift(sym, setTransitions.get(sym)));
        }
      }
    }
    for (Tuple3<ItemSet, Symbol, Action> action : extraActions) {
      addAction(grammar, actions, conflicts, action.first, action.second, action.third);
    }
    return new TransitionTable(goal, map, actions, conflicts);
  }

  private static void addAction(Grammar grammar,
      Map<ItemSet, Map<Symbol, Action>> actions,
      Collection<Conflict> conflicts,
      ItemSet set, Symbol sym, Action action) {
    Map<Symbol, Action> setActions = actions.get(set);
    if (setActions.containsKey(sym)) {
      Action otherAction = setActions.get(sym);
      SelectedAction selected = selectAction(action, otherAction, grammar);
      ActionKind kind = action.kind();
      ActionKind otherKind = otherAction.kind();
      if (selected == SelectedAction.SHIFT) {
        if (kind == ActionKind.SHIFT && otherKind == ActionKind.REDUCE) {
          setActions.put(sym, action);
          return;
        } else if (kind == ActionKind.REDUCE && otherKind == ActionKind.SHIFT) {
          return;
        }
      }
      if (selected == SelectedAction.REDUCE) {
        if (kind == ActionKind.SHIFT && otherKind == ActionKind.REDUCE) {
          return;
        } else if (kind == ActionKind.REDUCE && otherKind == ActionKind.SHIFT) {
          setActions.put(sym, action);
          return;
        }
      }
      conflicts.add(new Conflict(set, sym, action, setActions.get(sym)));
    } else {
      setActions.put(sym, action);
    }
  }

  enum SelectedAction {
    SHIFT, REDUCE, NONE
  }

  public static SelectedAction selectAction(Action firstAction, Action secondAction,
      Grammar grammar) {
    String first = firstToken(firstAction);
    String second = firstToken(secondAction);
    if (grammar.nonassoc(first) || grammar.nonassoc(second)) {
      // Nonassociative tokens may not be part in any conflict.
      return SelectedAction.NONE;
    }
    if (grammar.precedence(first) < grammar.precedence(second)) {
      return SelectedAction.SHIFT;
    } else if (grammar.precedence(first) > grammar.precedence(second)) {
      return SelectedAction.REDUCE;
    } else if (grammar.left(first) && grammar.left(second)) {
      return SelectedAction.REDUCE;
    } else if (grammar.right(first) && grammar.right(second)) {
      return SelectedAction.SHIFT;
    } else {
      if (firstAction.kind() == ActionKind.SHIFT && secondAction.kind() == ActionKind.REDUCE
          || firstAction.kind() == ActionKind.REDUCE && secondAction.kind() == ActionKind.SHIFT) {
        // TODO: improve shift/reduce handling (add option to convert warning to error).
        System.err.println("Warning: resolved SHIFT/REDUCE conflict by selecting SHIFT:");
        System.err.println(firstAction);
        System.err.println(secondAction);
        return SelectedAction.SHIFT;
      }
      return SelectedAction.NONE;
    }
  }

  private static String firstToken(Action action) {
    if (action instanceof Shift) {
      return ((Shift) action).sym.name();
    } else if (action instanceof Reduce) {
      for (Symbol sym : ((Reduce) action).rule.rhs) {
        if (sym.isTerminal()) {
          return sym.name();
        }
      }
    }
    return "?sym";
  }

  public void printTables(Grammar grammar, List<ItemSet> itemSets) {
    List<Symbol> nonterminals = new ArrayList<>();
    List<Symbol> terminals = new ArrayList<>();
    List<Conflict> conflicts = new ArrayList<>();
    for (Symbol sym : grammar.syms) {
      if (sym.isTerminal()) {
        terminals.add(sym);
      } else {
        nonterminals.add(sym);
      }
    }

    System.out.println("Transition Table");
    System.out.println("----------------");
    for (Symbol sym : grammar.syms) {
      System.out.print("\t" + sym + ",");
    }
    System.out.println();
    for (ItemSet set : itemSets) {
      Map<Symbol, ItemSet> transitions = map.get(set);
      System.out.print("S" + set.id());
      for (Symbol sym : grammar.syms) {
        if (transitions.containsKey(sym)) {
          System.out.print("\tS" + transitions.get(sym).id() + ",");
        } else {
          System.out.print("\t,");
        }
      }
      System.out.println();
    }
    System.out.println();
    System.out.println("Action Table");
    System.out.println("------------");
    for (Symbol sym : terminals) {
      System.out.print("\t" + sym + ",");
    }
    System.out.println();
    for (ItemSet set : itemSets) {
      System.out.print("S" + set.id());
      Map<Symbol, Action> setActions = actions.get(set);
      for (Symbol sym : terminals) {
        if (setActions.containsKey(sym)) {
          Action action = setActions.get(sym);
          System.out.print("\t" + action.code() + ",");
        } else {
          System.out.print("\t,");
        }
      }
      System.out.println();
    }
    System.out.println();
    System.out.println("Goto Table");
    System.out.println("----------");
    for (Symbol sym : nonterminals) {
      System.out.print("\t" + sym + ",");
    }
    System.out.println();
    for (ItemSet set : itemSets) {
      Map<Symbol, ItemSet> transitions = map.get(set);
      System.out.print("S" + set.id());
      for (Symbol sym : nonterminals) {
        if (transitions.containsKey(sym)) {
          System.out.print("\tS" + transitions.get(sym).id() + ",");
        } else {
          System.out.print("\t,");
        }
      }
      System.out.println();
    }
  }

  public void printConflicts(Grammar grammar) {
    Map<ItemSet, Collection<Conflict>> map = new HashMap<>();
    for (Conflict conflict : conflicts) {
      if (!map.containsKey(conflict.set)) {
        map.put(conflict.set, new ArrayList<Conflict>());
      }
      map.get(conflict.set).add(conflict);
    }
    for (Map.Entry<ItemSet, Collection<Conflict>> entry : map.entrySet()) {
      System.err.println("Conflicts:");
      for (Conflict conflict : entry.getValue()) {
        System.err.print("  ");
        System.err.println(conflict);
      }
      System.err.format("Context: %s%n", entry.getKey());
    }
    for (Conflict conflict : conflicts) {
      if (grammar.nonassoc(conflict.sym.name())) {
        throw new Error(String.format(
            "Symbol %s is nonassociative, but there are conflicts including the symbol.",
            conflict.sym.name()));
      }
      if (conflict.action instanceof Reduce && conflict.otherAction instanceof Reduce) {
        throw new Error("Parser contains reduce-reduce conflict.");
      }
    }
  }
}
