/* Copyright (c) 2017, Jesper Öqvist
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. Neither the name of the copyright holder nor the names of its
 * contributors may be used to endorse or promote products derived from this
 * software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */
package org.extendj.neobeaver;

import org.extendj.Trace;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Stack;

/**
 * Parser for beaver parser specifications.
 */
public class BeaverParser {
  public static class SpecError extends Error {
    public SpecError(String message) {
      super(message);
    }
  }
  public static Grammar buildParser(Trace trace, Parser.Symbol parseTree) {
    Symbol goalSym;
    List<Rule> rules;
    Stack<Parser.Symbol> stack;
    Map<String, String> typeMap;
    SymbolCache cache;
    Set<String> left;
    Set<String> right;
    Set<String> nonassoc;
    Set<String> assoc;
    Map<String, Integer> precedence;
    Set<Symbol> terminals;
    Set<String> terminalNames;
    String packageName;
    String className;
    String header;
    String embed;


    // TODO(joqvist): make this more flexible to changes in the parsing grammar.
    try {
      trace.pushEvent("parse");

      goalSym = Symbol.EOF;
      rules = new ArrayList<>();
      stack = new Stack<>();
      typeMap = new HashMap<>();
      cache = new SymbolCache();
      left = new HashSet<>();
      right = new HashSet<>();
      nonassoc = new HashSet<>();
      assoc = new HashSet<>();
      precedence = new HashMap<>();
      terminals = new HashSet<>();
      terminalNames = new HashSet<>();

      try {
        trace.pushEvent("typeMap");
        // First, build the production type map and determine the goal symbol.
        stack.push(parseTree);
        while (!stack.isEmpty()) {
          Parser.Symbol top = stack.pop();
          if (top instanceof Parser.Nonterminal) {
            Parser.Nonterminal nta = (Parser.Nonterminal) top;
            switch (nta.name) {
              case "productionList":
                if (nta.children.length > 1) {
                  stack.push(nta.children[1]);
                }
                stack.push(nta.children[0]);
                break;
              case "directive":
                String directive = literal(nta.children[0]);
                switch (directive) {
                  case "%class":
                  case "%package":
                  case "%header":
                  case "%embed":
                  case "%left":
                  case "%right":
                  case "%nonassoc":
                  case "%terminals":
                    break;
                  case "%typeof": {
                    // TODO: handle type declaration errors.
                    addType(cache, typeMap, nta, literal(nta.children[1]),
                        literal(nta.children[3]));
                    break;
                  }
                  case "%goal":
                    if (goalSym == Symbol.EOF) {
                      goalSym = cache.nta(literal(nta.children[1]));
                      goalSym.setPosition(nta.getPosition());
                    } else {
                      System.err.format("WARNING: %s multiple goal symbols specified.%n",
                          nta.getPosition());
                    }
                    break;
                  default:
                    throw new Error("Unknown directive: " + directive);
                }
                break;
              case "production":
                // TODO(joqvist): handle conflicting production types for the same NTA.
                String name = literal(nta.children[1]);
                Parser.Symbol type = nta.children[0];
                if (!(type instanceof Parser.EmptyOpt)) {
                  addType(cache, typeMap, nta, name, literal(type));
                }
                Symbol lhs;
                if (cache.contains(name)) {
                  lhs = cache.get(name);
                } else {
                  lhs = cache.nta(name);
                  lhs.setPosition(nta.getPosition());
                }
                lhs.setPosition(nta.getPosition());
                if (goalSym == Symbol.EOF) {
                  goalSym = lhs;
                }
                break;
              default:
                throw new Error("Unknown production: " + nta.name);
            }
          }
        }
      } finally {
        trace.popEvent();
      }

      try {
        trace.pushEvent("productions");
        int nextPrecedence = 1;

        // Now we build all productions.
        stack.push(parseTree);
        while (!stack.isEmpty()) {
          Parser.Symbol top = stack.pop();
          if (top instanceof Parser.Nonterminal) {
            Parser.Nonterminal nta = (Parser.Nonterminal) top;
            switch (nta.name) {
              case "productionList":
                if (nta.children.length > 1) {
                  stack.push(nta.children[1]);
                }
                stack.push(nta.children[0]);
                break;
              case "directive":
                String directive = literal(nta.children[0]);
                switch (directive) {
                  case "%class":
                  case "%package":
                  case "%header":
                  case "%embed":
                    break;
                  case "%left":
                    nextPrecedence = parseAssocRules(nta, assoc, left, precedence, nextPrecedence);
                    break;
                  case "%right":
                    nextPrecedence = parseAssocRules(nta, assoc, right, precedence, nextPrecedence);
                    break;
                  case "%nonassoc":
                    nextPrecedence =
                        parseAssocRules(nta, assoc, nonassoc, precedence, nextPrecedence);
                    break;
                  case "%terminals":
                    parseTerminals(cache, nta, terminals, terminalNames);
                    break;
                  case "%typeof":
                  case "%goal":
                    break;
                  default:
                    throw new Error("Unknown directive: " + directive);
                }
                break;
              case "production":
                String name = literal(nta.children[1]);
                String lhsType;
                if (typeMap.containsKey(name)) {
                  lhsType = typeMap.get(name);
                } else {
                  lhsType = "Symbol";
                }
                Symbol lhs;
                if (cache.contains(name)) {
                  lhs = cache.get(name);
                } else {
                  lhs = cache.nta(name);
                  lhs.setPosition(nta.getPosition());
                }
                Stack<Parser.Symbol> rstack = new Stack<>();
                rstack.push(nta.children[3]);
                while (!rstack.isEmpty()) {
                  Parser.Symbol rtop = rstack.pop();
                  if (rtop instanceof Parser.Nonterminal) {
                    Parser.Nonterminal rnta = (Parser.Nonterminal) rtop;
                    switch (rnta.name) {
                      case "ruleList":
                        rstack.push(rnta.children[0]);
                        if (rnta.children.length > 2) {
                          rstack.push(rnta.children[2]);
                        }
                        break;
                      case "rule":
                        List<Symbol> rhs = new ArrayList<>();
                        List<String> names = new ArrayList<>();
                        parseComponents(cache, rnta.children[0], rhs, names);
                        if (rnta.children.length > 1) {
                          // This rule has a semantic action.
                          String action = parseAction(rnta.children[1]);
                          rules.add(new Rule(-1, lhs, rhs, action, names, lhsType));
                        } else {
                          // No semantic action.
                          rules.add(new Rule(-1, lhs, rhs, "", names, lhsType));
                        }
                        break;
                      default:
                        throw new Error("Unknown production rule: " + rnta.name);
                    }
                  } else {
                    throw new Error("Expected NTA.");
                  }
                }
                break;
              default:
                throw new Error("Unknown production: " + nta.name);
            }
          }
        }
      } finally {
        trace.popEvent();
      }
      try {
        trace.pushEvent("packageName");
        packageName = parsePackageName(parseTree);
      } finally {
        trace.popEvent();
      }
      try {
        trace.pushEvent("className");
        className = parseClassName(parseTree);
      } finally {
        trace.popEvent();
      }
      try {
        trace.pushEvent("header");
        header = parseHeader(parseTree);
      } finally {
        trace.popEvent();
      }
      try {
        trace.pushEvent("embed");
        embed = parseEmbed(parseTree);
      } finally {
        trace.popEvent();
      }
    } finally {
      trace.popEvent();
    }
    try {
      trace.pushEvent("Grammar.build");
      return Grammar.build(trace, rules, terminals, goalSym, embed,
          header, className, packageName,
          left, right, nonassoc, precedence);
    } finally {
      trace.popEvent();
    }
  }

  private static void addType(SymbolCache cache, Map<String, String> typeMap,
      Parser.Nonterminal nta, String name,
      String type) {
    if (typeMap.containsKey(name) && !typeMap.get(name).equals(type)) {
      System.err.format("%s: WARNING: conflicting type declarations for %s: "
              + "new type %s is not same as previous type %s%n",
          nta.getPosition(),
          name, type, typeMap.get(name));
    }
    typeMap.put(name, type);
  }

  private static int parseAssocRules(Parser.Nonterminal nta, Set<String> assoc, Set<String> left,
      Map<String, Integer> precedence, int nextPrecedence) {
    Stack<Parser.Symbol> stack = new Stack<>();
    stack.push(nta.children[1]);
    while (!stack.isEmpty()) {
      Parser.Symbol top = stack.pop();
      if (top instanceof Parser.Nonterminal) {
        Parser.Nonterminal rnta = (Parser.Nonterminal) top;
        switch (rnta.name) {
          case "symList":
            stack.push(rnta.children[0]);
            if (rnta.children.length > 2) {
              stack.push(rnta.children[2]);
            }
            break;
          default:
            throw new Error("Unknown production rule: " + rnta.name);
        }
      } else {
        String name = literal(top);
        if (assoc.contains(name)) {
          System.err.format("WARNING: associativity already specified for %s%n", name);
        } else {
          precedence.put(name, nextPrecedence);
          assoc.add(name);
          left.add(name);
        }
      }
    }
    return nextPrecedence + 1;
  }

  private static void parseTerminals(SymbolCache cache,
      Parser.Nonterminal nta, Set<Symbol> terminals,
      Set<String> terminalNames) {
    Stack<Parser.Symbol> stack = new Stack<>();
    stack.push(nta.children[1]);
    while (!stack.isEmpty()) {
      Parser.Symbol top = stack.pop();
      if (top instanceof Parser.Nonterminal) {
        Parser.Nonterminal rnta = (Parser.Nonterminal) top;
        switch (rnta.name) {
          case "symList":
            stack.push(rnta.children[0]);
            if (rnta.children.length > 2) {
              stack.push(rnta.children[2]);
            }
            break;
          default:
            throw new Error("Unknown production rule: " + rnta.name);
        }
      } else {
        String name = literal(top);
        if (terminalNames.contains(name)) {
          System.err.format("WARNING: duplicate terminal declared: %s%n", name);
        } else {
          Symbol term = cache.terminal(name);
          term.setPosition(top.getPosition());
          terminals.add(term);
          terminalNames.add(name);
        }
      }
    }
  }

  private static String parseAction(Parser.Symbol child) {
    if (child instanceof Parser.Nonterminal) {
      Parser.Nonterminal nta = (Parser.Nonterminal) child;
      switch (nta.name) {
        case "semanticAction":
          return literal(nta.children[1]).trim();
        default:
          throw new Error("Unexpected action type: " + nta.name);
      }
    } else {
      throw new Error("Expected NTA, but found: " + child);
    }
  }

  private static void parseComponents(SymbolCache cache, Parser.Symbol child,
      List<Symbol> rhs, List<String> names) {
    Stack<Parser.Symbol> stack = new Stack<>();
    stack.push(child);
    Set<String> usedNames = new HashSet<>();
    while (!stack.isEmpty()) {
      Parser.Symbol top = stack.pop();
      if (top instanceof Parser.Nonterminal) {
        Parser.Nonterminal nta = (Parser.Nonterminal) top;
        switch (nta.name) {
          case "componentList":
            if (nta.children.length > 1) {
              stack.push(nta.children[1]);
            }
            if (nta.children.length > 0) {
              stack.push(nta.children[0]);
            }
            break;
          case "precedence": // TODO(joqvist): implement this.
            break;
          case "component":
            Tuple<Symbol, String> sym = parseSymbol(cache, nta.children[0]);
            if (nta.children.length == 2) {
              String quantifier = literal(nta.children[1]);
              switch (quantifier) {
                case "*":
                  rhs.add(ListComponent.buildOptional(sym.first, cache));
                  break;
                case "+":
                  rhs.add(ListComponent.build(sym.first, cache));
                  break;
                case "?":
                  rhs.add(OptionalComponent.build(sym.first, cache));
                  break;
                default:
                  throw new Error("Unknown quantifier: " + quantifier);
              }
            } else {
              rhs.add(sym.first);
            }
            if (usedNames.contains(sym.second)) {
              throw new SpecError(String.format(
                  "%s: can not reuse component name \"%s\" in this production!",
                  nta.getPosition(), sym.second));
            }
            usedNames.add(sym.second);
            names.add(sym.second);
            break;
          default:
            throw new Error("Unknown right-hand-side component: " + nta.name);
        }
      }
    }
  }

  /**
   * @return a tuple containing the symbol and action name of a production component.
   */
  private static Tuple<Symbol, String> parseSymbol(SymbolCache cache, Parser.Symbol sym) {
    if (sym instanceof Parser.Nonterminal) {
      Parser.Nonterminal nta = (Parser.Nonterminal) sym;
      switch (nta.name) {
        case "symbol":
          String name = literal(nta.children[0]);
          String actionName;
          if (nta.children.length == 3) {
            actionName = literal(nta.children[2]);
          } else {
            actionName = name;
          }
          if (cache.contains(name)) {
            return new Tuple<>(cache.get(name), actionName);
          } else {
            Symbol tok = cache.terminal(name);
            tok.setPosition(nta.getPosition());
            return new Tuple<>(tok, actionName);
          }
        default:
          throw new Error("Expected symbol production.");
      }
    } else {
      throw new Error("Can't convert non-NTA to symbol.");
    }
  }

  private static String parseEmbed(Parser.Symbol tree) {
    Stack<Parser.Symbol> stack = new Stack<>();
    stack.push(tree);
    while (!stack.isEmpty()) {
      Parser.Symbol top = stack.pop();
      if (top instanceof Parser.Nonterminal) {
        Parser.Nonterminal nta = (Parser.Nonterminal) top;
        switch (nta.name) {
          case "productionList":
            if (nta.children.length > 1) {
              stack.push(nta.children[1]);
            }
            stack.push(nta.children[0]);
            break;
          case "directive":
            String directive = literal(nta.children[0]);
            switch (directive) {
              case "%embed":
                return literal(nta.children[2]);
              default:
                break;
            }
            break;
        }
      }
    }
    return "";
  }

  private static String parseHeader(Parser.Symbol tree) {
    Stack<Parser.Symbol> stack = new Stack<>();
    stack.push(tree);
    while (!stack.isEmpty()) {
      Parser.Symbol top = stack.pop();
      if (top instanceof Parser.Nonterminal) {
        Parser.Nonterminal nta = (Parser.Nonterminal) top;
        switch (nta.name) {
          case "productionList":
            if (nta.children.length > 1) {
              stack.push(nta.children[1]);
            }
            stack.push(nta.children[0]);
            break;
          case "directive":
            String directive = literal(nta.children[0]);
            switch (directive) {
              case "%header":
                return literal(nta.children[2]);
              default:
                break;
            }
            break;
        }
      }
    }
    return "";
  }

  private static String parseClassName(Parser.Symbol tree) {
    Stack<Parser.Symbol> stack = new Stack<>();
    stack.push(tree);
    while (!stack.isEmpty()) {
      Parser.Symbol top = stack.pop();
      if (top instanceof Parser.Nonterminal) {
        Parser.Nonterminal nta = (Parser.Nonterminal) top;
        switch (nta.name) {
          case "productionList":
            if (nta.children.length > 1) {
              stack.push(nta.children[1]);
            }
            stack.push(nta.children[0]);
            break;
          case "directive":
            String directive = literal(nta.children[0]);
            switch (directive) {
              case "%class":
                return literal(nta.children[1]).trim();
              default:
                break;
            }
            break;
        }
      }
    }
    return "Parser";
  }

  private static String parsePackageName(Parser.Symbol tree) {
    Stack<Parser.Symbol> stack = new Stack<>();
    stack.push(tree);
    while (!stack.isEmpty()) {
      Parser.Symbol top = stack.pop();
      if (top instanceof Parser.Nonterminal) {
        Parser.Nonterminal nta = (Parser.Nonterminal) top;
        switch (nta.name) {
          case "productionList":
            if (nta.children.length > 1) {
              stack.push(nta.children[1]);
            }
            stack.push(nta.children[0]);
            break;
          case "directive":
            String directive = literal(nta.children[0]);
            switch (directive) {
              case "%package":
                return literal(nta.children[1]).trim();
              default:
                break;
            }
            break;
        }
      }
    }
    return "";
  }

  /**
   * Takes a symbol that is a token and returns its literal value.
   *
   * <p>The caller should ensure that the symbol is in fact a token.
   */
  private static String literal(Parser.Symbol child) {
    return ((Parser.Token) child).literal;
  }
}
