/**
 * Simple modal window
 * This class is intended for uses where modalbox may seem overkill
 */
loom.ui.Modal = Class.create({
	
	initialize: function(options) {
		this.options = Object.extend({
			lockScreen: false, // true to lock the interface so the user has to interact with the modal
			overlayOpacity: .5 // the opacity of the modal overlay
		}, options || {});
		this.onClick = this.onClick.bindAsEventListener(this);
		this.shim = this.createIfNotExists('iframe', { 
			id: 'modalShim', 
			style: 'display: none', 
			src: 'javascript:false;', 
			frameborder: 0, 
			scrolling: "no"
		});
		this.overlay = this.createIfNotExists('div', { id: 'modalOverlay', style: 'display: none; opacity: 0' });
		this.dialog = this.createIfNotExists('div', { id: 'modalDialog', style: 'display:none' });
	},
	
	createIfNotExists: function(element, options) {
		var e = $(options.id);
		if (!e) {
			e =  new Element(element, options);
			$(document.body).insert(e);
		}
		return e;
	},
	
	/**
	 * Display the modal window
	 * @param html the HTML (string) or Element to put inside the window container
	 */
	show: function(html) {
		this.dialog.update(html);
		this.overlay.observe('click', this.onClick);
		this.overlay.appear({ 
			to: this.options.overlayOpacity, 
			afterFinish: function() {
				this.dialog.show();
				// ie6 = absolute, other browsers = fixed
				var offsets = this.dialog.getStyle('position') == 'absolute'? document.viewport.getScrollOffsets() : { top: 0, left: 0 };
				this.dialog.setStyle({ 
					top: (offsets.top + ((document.viewport.getHeight() - this.dialog.getHeight()) / 2).floor()) + 'px', 
					left: (offsets.left + ((document.viewport.getWidth() - this.dialog.getWidth()) / 2).floor()) + 'px'
				});
				this.shim.setStyle({
					top: this.dialog.getStyle('top'),
					left: this.dialog.getStyle('left'),
					width: this.dialog.getWidth() + 'px', 
					height: this.dialog.getHeight() + 'px'
				});
				this.shim.show();
				this.dialog.fire('modal:show'); 
			}.bind(this)
		});
	},
	
	hide: function(event) {
		var afterFinish = function() {
			this.overlay.hide();
			this.overlay.stopObserving('click', this.onClick);
			this.dialog.fire('modal:hide');
		}.bind(this);
		this.dialog.setStyle({ 
			top: '-10000px', 
			left: '-10000px'
		});
		this.dialog.hide();
		this.shim.hide();
		this.overlay.fade({ afterFinish: afterFinish });
	},
	
	/**
	 * Default onclick handler: close if the user clicks outside the modal
	 */
	onClick: function(event) {
		if (!event.findElement('#modalDialog')) { // if clicked outside the modal
			!this.options.lockScreen && this.hide(event); // close the modal
			event.stop();
		}
	}
	
});
