/*
 * Copyright 2002-2006 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.loom.addons.autocompleter;

import java.util.Map;

import org.loom.binding.AnnotationContainer;
import org.loom.converter.AnnotationDrivenConverterFactory;
import org.loom.converter.Converter;
import org.loom.exception.ConfigException;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.stereotype.Component;

@Component
public class AutocompletedConverterFactory implements AnnotationDrivenConverterFactory<Autocompleted>, ApplicationContextAware {

	private ApplicationContext applicationContext;
	
	public Converter getConverter(AnnotationContainer propertyWrapper, Autocompleted annotation) {
		
		AbstractAutocompletedConverter converter;
		
		if (annotation.separatorTokens().length() > 0) {
			converter = getMultipleValueConverter();
			((MultipleAutocompletedConverter)converter).setSeparatorTokens(annotation.separatorTokens());
			int length = propertyWrapper.getItemMetadataSize();
			Class clazz = length == 0? propertyWrapper.getPropertyClass() : propertyWrapper.getItemMetadata(length - 1).getItemClass();
			converter.setPersistentClass(clazz);
		} else {
			converter = getSingleValueConverter();
			converter.setPersistentClass(propertyWrapper.getPropertyClass());
		}
		
		converter.setConvertedClass(propertyWrapper.getPropertyClass());
		if (annotation.query().length() > 0) {
			converter.setQuery(annotation.query());
		}
		converter.setPropertyName(annotation.propertyName());
		return converter;
	}

	protected MultipleAutocompletedConverter getMultipleValueConverter() {
		return createConverter(MultipleAutocompletedConverter.class);
	}

	protected AutocompletedConverter getSingleValueConverter() {
		return createConverter(AutocompletedConverter.class);
	}

	@SuppressWarnings("unchecked")
	private <T> T createConverter(Class<T> converterClass) {
		Map m = applicationContext.getBeansOfType(converterClass);
		if (m.isEmpty()) {
			throw new ConfigException("No converter with type " + converterClass.getName() + " has been found");
		}
		return (T) m.values().iterator().next();
	}

	public void setApplicationContext(ApplicationContext applicationContext) {
		this.applicationContext = applicationContext;
	}

	public Class<Autocompleted> getAnnotationClass() {
		return Autocompleted.class;
	}
	
}
