/*
 * Copyright 2002-2006 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.loom.addons.rating;

import java.io.IOException;
import java.math.BigDecimal;
import java.math.RoundingMode;

import javax.servlet.jsp.JspException;

import org.loom.config.ConfigProperties;
import org.loom.i18n.Message;
import org.loom.servlet.names.HtmlExtendedAttributeNames;
import org.loom.tags.StringWriterWrapper;
import org.loom.tags.core.InputSelectTag;
import org.loom.tags.core.NumberFieldTag;
import org.loom.tags.core.Option;
import org.loom.tags.core.ValueComparator;
import org.tldgen.annotations.Attribute;
import org.tldgen.annotations.BodyContent;
import org.tldgen.annotations.Tag;

/**
 * Shows a rating star component with the current rating and allow the user to 
 * cast a vote. 
 * <p>
 * This tag can be used to display a rating result by using its total 
 * and numVotes attributes or a concrete vote using its name attribute. 
 * If the latest, an input field will be rendered.
 * </p>
 * <p>
 * It is responsibility of the application developer to disable this tag if the user has already voted. The
 * javascript library will disable the voting system only once, if the user revisits the page a new vote
 * could be sent.
 * </p>
 * @author Ignacio Coloma
 * @author German Garcia
 */
@Tag(
		dynamicAttributes=true,
		bodyContent=BodyContent.SCRIPTLESS,
		example=
			"&lt;a:inputRating id=\"vote\" name=\"vote\" total=\"${action.total}\" numVotes=\"${action.numVotes}\"/>\n" +
			"&lt;script>new loom.ui.Rating($('vote'));&lt;/script>"
	)
public class InputRatingTag extends InputSelectTag implements NumberFieldTag {

	private static String CSS = ConfigProperties.getInstance().getProperty("css.RatingTag", "rating"); 
 
	/** number of decimals included in the average value */
	private static final int AVERAGE_SCALE = 1;

	/** the aggregated sum of all votes, null if none */
	@Attribute
	private BigDecimal total;
	
	/** 
	 * The number of votes received, null for none.
	 */
	@Attribute
	private Integer numVotes;
	
	private BigDecimal max;
	
	private BigDecimal min;
	
	/**
	 * Set to true to make the minimum vote value exclusive (default false)
	 */
	@Attribute
	private boolean excludeMin = false;
	
	/**
	 * Set to true to make the maximum vote value inclusive (default true)
	 */
	@Attribute
	private boolean excludeMax = true;
	
	/** the number of decimal  digits to apply when calculating the average, default 1 */
	@Attribute
	private Integer scale = AVERAGE_SCALE;
	
	/** average rating. This field is calculated */
	private BigDecimal average;
	
	/** true to render the current rating value as text, default true */
	@Attribute
	private boolean renderRatingText = true;
	
	public InputRatingTag() {
		setCssClass(CSS);
		setOptionsComparatorClass(ValueComparator.class.getName());
		setAddTranslatedName(false);
		setEscapeHTML(false);
	}

	@Override
	public void initTag() {
		super.initTag();
		
		// if only total has been specified, suppose 1 vote
		if (total != null && numVotes == null) { 
			setNumVotes(1);
		}

		// calculate option list automatically
		if (max !=  null) {
			int imax = excludeMax? max.intValue() : max.intValue() + 1;
			int imin = min == null? 0 : min.intValue();
			imin = !excludeMin? imin : min.intValue() + 1;
			for (int i = imin; i <= imax; i++) {
				addOption(new Option(i, isTranslateLabels()? getName() + '.' + i : String.valueOf(i), isTranslateLabels()));
			}
		} 
		
		// calculate average
		if (total != null && numVotes != null && numVotes > 0) {
			this.average = total.divide(new BigDecimal(numVotes), scale, RoundingMode.HALF_DOWN);
		}
		setExtendedAttribute("average", average);
		if (scale != AVERAGE_SCALE) {
			setExtendedAttribute("scale", scale);
		}
		
	}
	
	/**
	 * If name == null, set to render as text (as a div)
	 */
	@Override
	public boolean shouldRenderAsText() {
		return getName() == null || super.shouldRenderAsText();
	}
	
	@Override
	protected void printParentAttributes() throws IOException, JspException {
		if (min !=  null) {
			int imin = min.intValue();
			out.printAttribute(HtmlExtendedAttributeNames.MIN, excludeMin? imin + 1 : imin);
		}
		if (max !=  null) {
			int imax = max.intValue();
			out.printAttribute(HtmlExtendedAttributeNames.MAX, excludeMax? imax : imax + 1);
		}
		
		super.printParentAttributes();
	}
	
	@Override
	public boolean shouldRenderLabel() {
		return getName() != null && super.shouldRenderLabel();
	}
	
	@Override
	public void doTagImpl() throws JspException, IOException {
		super.doTagImpl();
		renderMessage();
	}
	
	@Override
	public String renderAsText() {
		try {
			this.out = new StringWriterWrapper();
			out.print("<span class=\"rating-label\">");
			out.print("<span");
			printParentAttributes();
			out.print("> </span>");
			renderMessage();
			out.print("</span>");
			return ((StringWriterWrapper)out).getContents();
		} catch (IOException e) {
			throw new RuntimeException(e);
		} catch (JspException e) {
			throw new RuntimeException(e);
		}
	}
	
	protected void renderMessage() throws IOException {
		// the current results
		if (renderRatingText && average != null) {
			out.print("<span class=\"message\">");
			Message m = new Message("loom.rating.message")
			.addArg("average", average)
			.addArg("total", total)
			.addArg("numVotes", numVotes);
			out.print(getMessagesRepository().translateMessage(m));
			out.print("</span>");
		}
	}

	public void setTotal(BigDecimal total) {
		this.total = total;
	}
	
	public void setNumVotes(Integer numVotes) {
		this.numVotes = numVotes;
		setExtendedAttribute("numVotes", numVotes);
	}

	public void setRenderRatingText(boolean renderText) {
		this.renderRatingText = renderText;
	}

	public BigDecimal getTotal() {
		return total;
	}

	public Integer getNumVotes() {
		return numVotes;
	}

	public boolean isRenderRatingText() {
		return renderRatingText;
	}

	public BigDecimal getAverage() {
		return average;
	}

	public BigDecimal getMax() {
		return max;
	}

	public void setMax(BigDecimal max) {
		this.max = max;
	}
	
	/** 
	 * The maximum vote value. 
	 * If this value is specified, the list of options will be automatically calculated. 
	 * <p>
	 * If there is a NumberValidation annotation, this field is automatically calculated
	 * </p>
	 */
	// a double setter is needed because dumb JSP compilers do not know how to handle a BigDecimal 
	@Attribute
	public void setMaxValue(double max) {
		this.max = new BigDecimal(String.valueOf(max));
	}

	public BigDecimal getMin() {
		return min;
	}

	public void setMin(BigDecimal min) {
		this.min = min;
	}
	
	/** 
	 * The minimum vote value. 
	 * If this value is specified, the list of options will be automatically calculated 
	 * <p>
	 * If there is a NumberValidation annotation, this field is automatically calculated
	 * </p>
	 */
	// a double setter is needed because dumb JSP compilers do not know how to handle a BigDecimal 
	@Attribute
	public void setMinValue(double min) {
		this.min = new BigDecimal(String.valueOf(min));
	}

	public void setExcludeMax(boolean excludeMax) {
		this.excludeMax = excludeMax;
	}

	public void setExcludeMin(boolean excludeMin) {
		this.excludeMin = excludeMin;
	}

	public void setPrecision(Integer precision) {
		// unused
	}

	/** 
	 * If unspecified, this tag will be used to display a read-only value
	 */
	@Override
	@Attribute
	public void setName(String name) {
		super.setName(name);
	}
	
	public void setScale(Integer scale) {
		this.scale = scale;
	}

	public boolean isExcludeMin() {
		return excludeMin;
	}

	public boolean isExcludeMax() {
		return excludeMax;
	}

	
	
}
