/*
 * Copyright 2002-2006 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.loom.addons.multiupload;

import static org.loom.addons.servlet.names.RequestParameterNames.UPLOADED_FILES_PREFIX;

import java.io.IOException;
import java.util.List;

import javax.servlet.jsp.JspException;

import org.apache.commons.lang.StringUtils;
import org.loom.addons.servlet.names.HtmlExtendedAttributeNames;
import org.loom.i18n.FileSizeFormatter;
import org.loom.mapping.ParsedAction;
import org.loom.persistence.file.PersistentFile;
import org.loom.tags.AbstractDecoratedTag;
import org.loom.tags.WriterWrapper;
import org.loom.tags.core.InputFileTag;
import org.tldgen.annotations.Attribute;
import org.tldgen.annotations.BodyContent;
import org.tldgen.annotations.Tag;



/**
 * This tag generates multiple input file components in the browser.
 * <p>
 * If javascript is enabled the user may generate file fields by hand, but with 
 * javascript switched off it will render several file fields from the server. 
 * </p>
 * @author icoloma
 */
@Tag(
		dynamicAttributes=true,
		bodyContent=BodyContent.EMPTY,
		example=
			"&lt;l:form>\n" +
			"  &lt;a:multiUpload name=\"attachments\"/>\n" +
			"&lt;/l:form>\n" +
			"&lt;script>\n" +
			"  new loom.ui.MultiUpload($('multiupload'));\n" +
			"&lt;/script>"
		)
public class MultiUploadTag extends AbstractDecoratedTag {

	/** the POST parameter name */
	@Attribute(required=true)
	private String name;
	
	/** true to add static HTML fields (consumes more bandwidth), defaults true */
	@Attribute
	private boolean accessible = true;
	
	/** maximum number of rows to display if accessible is true, default to 10 */
	@Attribute
	private Integer maxRows;
	
	private FileSizeFormatter fileSizeFormatter = new FileSizeFormatter();
	
	protected Object getValueAsObject() {
		ParsedAction parsedAction = request.getParsedAction();
		if (parsedAction == null) {
			return null;
		}
		return parsedAction.getPropertyAsObject(name); 
	}
	
	@Override
	@SuppressWarnings("unchecked")
	public void doTagImpl() throws JspException, IOException {
		
		fileSizeFormatter.setMessagesRepository(getMessagesRepository());
		request.getCurrentForm().setMultipart(true);
		if (maxRows == null && accessible) {
			maxRows = Integer.valueOf(10);
		}
		
		out.print("<div");
		out.printAttribute("class", "multiupload");
		printParentAttributes();
		out.print(">\n");
		
		// Render the existing file IDs
		List<PersistentFile> files = (List<PersistentFile>) getValueAsObject();
		int index = 0;
		if (files != null && files.size() > 0) {
			out.print("<ul>\n");
			for (PersistentFile file : files) {
				render(out, file, index++);
			}
			out.print("</ul>\n");
		}

		// render multiple input file fields
		// * this code must generate the same HTML done via javascript
		// * this code will only be displayed in the browser if javascript is off
		if (accessible) {
			out.print("<noscript>\n<ul>\n");
			InputFileTag file = new InputFileTag();
			file.setWriter(out);
			file.setForm(request.getCurrentForm());
			file.setRequest(request);
			for (int i = 0; i < maxRows; i++) {
				file.setName(name + '[' + (index + i) + ']');
				out.print("<li>");
				file.doTagImpl();
				out.print("</li>\n");
			}
			out.print("</ul>\n</noscript>\n");
		}
		out.print("</div>\n");
			
	}

	/**
	 * Render one (already uploaded) file
	 * @param file
	 * @throws IOException 
	 */
	protected void render(WriterWrapper<?> out, PersistentFile file, int index) throws IOException {
		// do not render transient files, as this is probably a form with validation errors
		// and  they are not stored anywhere. 
		if (file.getId() == null) {
			return;
		}
		
		String filename = file.getFilename();
		out.print("<li");
		String extension = StringUtils.substringAfterLast(filename, ".");
		if (extension !=  null) {
			out.printAttribute("class", "file-" + extension);
		}
		out.print(">\n");
		
		// add checkbox to include file
		out.print("<input type=\"checkbox\"");
		out.printAttribute("name", UPLOADED_FILES_PREFIX + name + '[' + index + ']');
		out.printAttribute("value", file.getId());
		out.printAttribute("class", "checkbox");
		out.print(" checked/>");
		
		out.print("<span>");
		out.print(filename);
		if (file.getFileSize() != null) {
			out.print(" (").print(fileSizeFormatter.format(file.getFileSize())).print(")");
		}
		out.print("</span>\n");
        out.print("</li>\n");	

	}

	public Integer getMaxRows() {
		return maxRows;
	}

	public void setMaxRows(Integer maxRows) {
		this.maxRows = maxRows;
	}

	public boolean isAccessible() {
		return accessible;
	}

	public void setAccessible(boolean accessible) {
		this.accessible = accessible;
	}

	public String getName() {
		return name;
	}

	public void setName(String name) {
		this.name = name;
		setExtendedAttribute(HtmlExtendedAttributeNames.MULTIUPLOAD_NAME, name);
	}

}
