package org.loom.addons.multiupload;

import static org.loom.addons.servlet.names.RequestParameterNames.UPLOADED_FILES_PREFIX;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.regex.Pattern;

import org.loom.i18n.Message;
import org.loom.interceptor.AbstractPropertyBoundInterceptor;
import org.loom.log.Log;
import org.loom.mapping.ParsedAction;
import org.loom.persistence.file.FileManager;
import org.loom.persistence.file.PersistentFile;
import org.loom.resolution.Resolution;
import org.loom.servlet.LoomServletRequest;
import org.loom.servlet.params.FileParameter;
import org.loom.servlet.params.ServletRequestParameters;
import org.springframework.transaction.annotation.Transactional;

import com.google.common.collect.Maps;
import com.google.common.collect.Sets;

public class MultiuploadInterceptorImpl extends AbstractPropertyBoundInterceptor implements MultiUploadInterceptor {

	private FileManager fileManager;
	
	/** maximum file size allowed, null for no limit */
	private Long maxFileSize;
	
	/** minimum  number of files in the collection, null for no limit */
	private Integer minFilesCount;
	
	/** maximum  number of files in the collection, null for no limit */
	private Integer maxFilesCount;
	
	/** expected file name pattern, null for none */
    private Pattern filenameMaskPattern;
    
    /** true to remove deleted files from the persistent storage, default true */
    private boolean deleteRemovedFiles = true;
    
	private static Log log = Log.getLog(MultiuploadInterceptorImpl.class);
	
	public Resolution beforeValidate(ParsedAction action) {
		String propertyPath = getPropertyPath();
		
		LoomServletRequest request = action.getRequest();
		ServletRequestParameters requestParameters = request.getRequestParameters();
		
		// list of new uploaded files
		Map<String, FileParameter> uploadedFiles = Maps.newHashMap();
		
		// list of existing files that will be kept (not removed)
		Set<String> keptFiles = Sets.newHashSet();
		
		// 1.- Validate all files (size and file name)
		for (Map.Entry<String, FileParameter> entry : requestParameters.getFileParametersEntrySet()) {
			String paramName = entry.getKey();
			if (paramName.startsWith(propertyPath)) {
				FileParameter value = entry.getValue();
				uploadedFiles.put(paramName, value);
				requestParameters.setAssigned(paramName);
				
				if (maxFileSize != null && value.getFileSize() > maxFileSize) {
					addErrorMessage(action, "loom.validation.fileSizeTooLarge", value);
				} else if (filenameMaskPattern != null && !filenameMaskPattern.matcher(value.getFilename()).matches()) {
					addErrorMessage(action, "loom.validation.fileNameFailed", value);
				} 
			} 
		}
		for (Map.Entry<String, String> entry : requestParameters.getStringParametersEntrySet()) {
			if (entry.getKey().startsWith(UPLOADED_FILES_PREFIX)) {
				keptFiles.add(entry.getValue());
			}
		}
		request.setAttribute("_uploadedFiles", uploadedFiles);
		request.setAttribute("_keptFiles", keptFiles);
		
		// 2.- Validate the number of files that will be after processing the request
		int size = keptFiles.size() + uploadedFiles.size();
		if (minFilesCount != null && minFilesCount > size) {
			addErrorMessage(action, "multiupload.validation.minFilesFailed", null);
		}
		if (maxFilesCount != null && maxFilesCount < size) {
			addErrorMessage(action, "multiupload.validation.maxFilesFailed", null);
		}
		return null;
	}
	
	@SuppressWarnings("unchecked")
	@Transactional
	public Resolution beforeExecute(ParsedAction action) {
		String propertyPath = getPropertyPath();
		LoomServletRequest request = action.getRequest();
		Map<String, FileParameter> uploadedFiles = (Map<String, FileParameter>) request.getAttribute("_uploadedFiles");
		Set<String> keptFiles = (Set<String>) request.getAttribute("_keptFiles");
		
		// 1.- Remove existing files that are not included in this form submission
		Collection<PersistentFile> files = (Collection<PersistentFile>) action.getPropertyAsObject(propertyPath);
		if (files == null) {
			files = new ArrayList<PersistentFile>();
			action.setPropertyAsObject(propertyPath, files);
		}
		for (Iterator<PersistentFile> i = files.iterator(); i.hasNext(); ) {
			PersistentFile file = i.next();
			if (file != null && file.getId() != null && !keptFiles.contains(file.getId().toString())) {
				log.trace("Removing persistent file with id=", file.getId(), " because it was not included in the current request");
				i.remove();
				if (deleteRemovedFiles) {
					fileManager.remove(file.getId());
				}
			} 
		}
		
		// 2.- If everything's ok, add new files uploaded with this form submission
		if (!action.hasAnyError()) {
			for (FileParameter uploadedFile : uploadedFiles.values()) {
				log.debug("Saving uploaded file to database ", uploadedFile.getFilename());
				files.add(fileManager.merge(uploadedFile));
			}
		}
		return null;
	}
	
	/**
	 * Adds a validation error message to one uploaded file 
	 */
	public void addErrorMessage(ParsedAction action, String messageKey, FileParameter value) {
		Message message = new Message(messageKey);
		message.addArg(Message.VALUE_ARG, value);
		message.addArg("interceptor", this);
		message.addTranslatedArg(Message.PROPERTY_PATH_ARG, getPropertyPath());
		action.getMessages().addMessage(message);
	}
	
	public void setFileManager(FileManager fileManager) {
		this.fileManager = fileManager;
	}

	public void setMaxFileSize(Long maxFileSize) {
		this.maxFileSize = maxFileSize;
	}

	public void setFilenameMaskPattern(Pattern filenameMaskPattern) {
		this.filenameMaskPattern = filenameMaskPattern;
	}

	public Long getMaxFileSize() {
		return maxFileSize;
	}

	public Pattern getFilenameMaskPattern() {
		return filenameMaskPattern;
	}

	public Integer getMinFilesCount() {
		return minFilesCount;
	}

	public void setMinFilesCount(Integer minFilesCount) {
		this.minFilesCount = minFilesCount;
	}

	public Integer getMaxFilesCount() {
		return maxFilesCount;
	}

	public void setMaxFilesCount(Integer maxFilesCount) {
		this.maxFilesCount = maxFilesCount;
	}

	public boolean isDeleteRemovedFiles() {
		return deleteRemovedFiles;
	}

	public void setDeleteRemovedFiles(boolean deleteRemovedFiles) {
		this.deleteRemovedFiles = deleteRemovedFiles;
	}
	
}
