package org.loom.addons.recaptcha;

import java.io.IOException;
import java.util.Map;
import java.util.Set;

import javax.inject.Inject;
import javax.servlet.jsp.JspException;

import org.loom.i18n.MessagesRepositoryFactory;
import org.loom.tags.AbstractTag;
import org.loom.tags.functions.Functions;
import org.loom.url.UrlBuilder;
import org.tldgen.annotations.Attribute;
import org.tldgen.annotations.BodyContent;
import org.tldgen.annotations.Tag;

import com.google.common.collect.Maps;
import com.google.common.collect.Sets;

/**
 * Render a <a href="http://recaptcha.net/">reCAPTCHA</a> challenge field.
 * <p>
 * This tag supports the <a href="http://www.google.com/recaptcha/apidocs/captcha/client#look-n-feel">L&F 
 * personalization options</a> available for reCAPTCHA widgets by creating a javascript RecaptchaOptions object. You 
 * can also get greater control by specifying this object yourself following the reCAPTCHA documentation instead of 
 * using the theme attributes available here. 
 * @author icoloma
 */
@Tag(
		bodyContent=BodyContent.EMPTY,
		example="&lt;recaptcha publicKey=\"my-public-key\"/>"
	)
public class RecaptchaTag extends AbstractTag {

	/** request-aware attribute with the ReCaptcha error message code */
	public static final String RECAPTCHA_ERROR_MESSAGE = "recaptcha-message";
	
	/** the recaptcha challenge parameter name */
	public static final String CHALLENGE_PARAM = "recaptcha_challenge_field";
	public static final String RESPONSE_PARAM = "recaptcha_response_field";
	
	public static final String HTTP_SERVER = "api.recaptcha.net";
	public static final String HTTPS_SERVER = "api-secure.recaptcha.net";
	
	/** true to generate noscript tag, default true */
	@Attribute
	private boolean noscript = true;
	
	/** the ReCaptcha public key to use */
	@Attribute(required=true)
	private String publicKey;
	
	/** the user-specified list of options */
	private Map<String, Object> options = Maps.newHashMap();
	
	/** the set of languages supported by recaptcha */
	private final static Set<String> SUPPORTED_LANGUAGES = Sets.newHashSet("en", "nl", "fr", "de", "pt", "ru", "es", "tr");
	
	/** the set of themes supported by recaptcha */
	private final static Set<String> SUPPORTED_THEMES = Sets.newHashSet("red" , "white" , "blackglass" , "clean" , "custom");
	
	@Inject
	private MessagesRepositoryFactory messagesRepositoryFactory;

	@Override
	public void doTagImpl() throws JspException, IOException {
		UrlBuilder url = request.createUrl();
		if (request.isSecure()) {
			url.setScheme(UrlBuilder.SECURE_SCHEME);
			url.setHostname(HTTPS_SERVER);
		} else {
			url.setScheme(UrlBuilder.INSECURE_SCHEME);
			url.setHostname(HTTP_SERVER);
		}
		url.setRawUrl("/challenge");
		url.add("k", publicKey);
		url.add("error", request.getAttribute(RECAPTCHA_ERROR_MESSAGE));
	
		if (!options.isEmpty()) {
			out.start("script");
			out.print("var RecaptchaOptions=").print(Functions.json(options)).print(";");
			out.end("script");
		}
		out.start("script", "src", url.getURL()).end("script");
	
		if (noscript) {
			out.start("noscript");
			url.setRawUrl("/noscript");
			out.start("iframe", "src", url.getURL(), "height", 300, "width", 500, "frameborder", 0);
			out.end("iframe");
			out.start("textarea", "name", CHALLENGE_PARAM, "rows", 3, "cols", 40).end("textarea");
			out.startEnd("input", "name", RESPONSE_PARAM, "type", "hidden", "value", "manual_challenge");
			out.end("noscript");
		}
 	}

	public void setNoscript(boolean noscript) {
		this.noscript = noscript;
	}

	public void setPublicKey(String publicKey) {
		this.publicKey = publicKey;
	}
	
	/**
	 * Defines which theme to use for reCAPTCHA. Possible values are 'red' | 'white' | 'blackglass' | 'clean' | 'custom'.
	 */
	@Attribute
	public void setTheme(String theme) {
		if (!SUPPORTED_THEMES.contains(theme)) {
			throw new IllegalArgumentException("Theme not supported: " + theme + ". Available themes are: " + SUPPORTED_THEMES);
		}
		options.put("theme", theme);
	}
	
	/**
	 * Which language is used in the interface for the pre-defined themes. If the language is not supported by reCAPTCHA it will 
	 * be converted to the default MessagesRepositoryFactory locale.
	 */
	@Attribute
	public void setLang(String lang) {
		if (!SUPPORTED_LANGUAGES.contains(lang)) {
			lang = messagesRepositoryFactory.getDefaultLocale().getLanguage();
		}
		options.put("lang", lang);
	}
	
	/**
	 * When using custom theming, this is a div element which contains the widget. See the reCAPTCHA custom theming section for how to use this.
	 */
	@Attribute
	public void setCustomThemeWidget(String customThemeWidget) {
		options.put("custom_theme_widget", customThemeWidget);
	}
	
	/**
	 * Sets a tabindex for the reCAPTCHA text box.
	 */
	@Attribute
	public void setTabIndex(int tabIndex) {
		options.put("tabindex", tabIndex);
	}
	
}
