/*
 * Copyright 2002-2006 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.loom.addons.multiupload;

import static org.loom.addons.servlet.names.RequestParameterNames.UPLOADED_FILES_PREFIX;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.regex.Pattern;

import org.apache.commons.lang.StringUtils;
import org.loom.exception.RequiredAttributeException;
import org.loom.i18n.Message;
import org.loom.i18n.Messages;
import org.loom.interceptor.AbstractPropertyBoundInterceptor;
import org.loom.interceptor.ConfigurableInterceptor;
import org.loom.interceptor.PopulateListener;
import org.loom.interceptor.RetrieveEntityInterceptor;
import org.loom.log.Log;
import org.loom.mapping.ActionMapping;
import org.loom.mapping.ParsedAction;
import org.loom.persistence.ExtendedEntityManager;
import org.loom.persistence.file.FileManager;
import org.loom.persistence.file.PersistentFile;
import org.loom.resolution.Resolution;
import org.loom.servlet.LoomServletRequest;
import org.loom.servlet.params.FileParameterValue;
import org.loom.servlet.params.ParameterValue;
import org.loom.util.RequestUtils;

/**
 * Store and merge the files uploaded in this form submission.
 * Uploaded files that do not pass the configured validations (size and file name)
 * are not accepted.
 * 
 * Note that this controller does not remove orphan files from the underlying storage. It 
 * is responsibility of the application developer to do so using cleanup threads or similar. 
 * @author Ignacio Coloma
 */
public class MultiUploadInterceptor extends AbstractPropertyBoundInterceptor implements PopulateListener, ConfigurableInterceptor {
	
	/** used to retrieve the multiple file container entity */ 
	private RetrieveEntityInterceptor retrieveEntityInterceptor;
	
	private FileManager fileManager;
	
	/** maximum file size allowed, null for no limit */
	private Long maxFileSize;
	
	/** expected file name pattern, null for none */
    private Pattern filenameMaskPattern;
	
	private static Log log = Log.getLog();
	
	public MultiUploadInterceptor() {
		retrieveEntityInterceptor = new RetrieveEntityInterceptor();
		setRequiresConfiguration(true);
	}
	
	public void validate() {
		if (fileManager == null) {
			throw new RequiredAttributeException("fileManager");
		}
		retrieveEntityInterceptor.validate();
	}

	@Override
	public void setPropertyPath(String propertyPath) {
		super.setPropertyPath(propertyPath);
		retrieveEntityInterceptor.setPropertyPath(StringUtils.isEmpty(propertyPath)? null : StringUtils.substringBeforeLast(propertyPath, "."));
	}
	
	@Override
	public void setActionMapping(ActionMapping mapping) {
		super.setActionMapping(mapping);
		retrieveEntityInterceptor.setActionMapping(mapping);
	}
	
	@SuppressWarnings("unchecked")
	public Resolution beforePopulate(ParsedAction action) {
		String propertyPath = getPropertyPath();
		
		// 1.- Retrieve the container entity
		retrieveEntityInterceptor.beforePopulate(action);

		LoomServletRequest request = action.getRequest();
			
		// 2.- Remove existing files that are not included in this form submission
		Set<String> existingFileIds = getExistingFileIds(request);
		Collection<PersistentFile> files = (Collection<PersistentFile>) getPropertyBinder().getValue(action.getAction(), propertyPath);
		if (files == null) {
			files = new ArrayList();
			getPropertyBinder().setValue(action.getAction(), files, propertyPath);
		}
		for (Iterator<PersistentFile> i = files.iterator(); i.hasNext(); ) {
			PersistentFile file = i.next();
			String fileId = file.getId().toString();
			if (!existingFileIds.contains(fileId)) {
				log.trace("Removing persistent file with id=", fileId, " because it was not included in the current request");
				i.remove();
			} 
		}
		
		// 3.- Validate all files (size and file name)
		for (ParameterValue v : request.getParameters().getValues()) {
			if (v instanceof FileParameterValue && v.getName().startsWith(propertyPath)) {
				FileParameterValue value = (FileParameterValue) v;
				if (maxFileSize != null && value.getFileSize() > maxFileSize) {
					addErrorMessage(action, "loom.validation.fileSizeTooLarge", value);
					value.setAssigned(true);
				} else if (filenameMaskPattern != null && !filenameMaskPattern.matcher(value.getFilename()).matches()) {
					addErrorMessage(action, "loom.validation.fileNameFailed", value);
					value.setAssigned(true);
				} 
			}
		}
		
		// 4.- If everything's ok, add new files uploaded with this form submission
		for (ParameterValue v : request.getParameters().getValues()) {
			if (v instanceof FileParameterValue && v.getName().startsWith(propertyPath)) {
				FileParameterValue value = (FileParameterValue) v;
				if (!action.hasAnyError()) {
					log.debug("Saving uploaded file to database ", value.getFilename());
					PersistentFile uploadedFile = createFile(value);
					PersistentFile persistedFile = fileManager.merge(uploadedFile);
					files.add(persistedFile);
					log.debug("File ", value.getFilename(), " saved with id= ", persistedFile.getId());
				}
				
				// do not assign it later in the lifecycle
				value.setAssigned(true);
			}
		}
	
		return null;
	}
	
	/**
	 * Transforms an uploaded file into a PersistentFile instance
	 */
	private PersistentFile createFile(FileParameterValue value) {
		try {
			PersistentFile file = new PersistentFile();
			file.setFilename(value.getFilename());
			file.setFileSize(value.getFileSize());
			file.setContentType(value.getContentType());
			file.setContentsAsStream(value.getStream());
			return file;
		} catch (IOException e) {
			throw new RuntimeException(e);
		}
	}

	/**
	 * Adds a validation error message to one uploaded file 
	 */
	public void addErrorMessage(ParsedAction action, String messageKey, FileParameterValue value) {
		Message message = new Message();
		message.setPropertyPath(value.getName());
		message.setMessageKey(messageKey);
		message.addArg(Messages.VALUE_ARG, value);
		message.addArg("interceptor", this);
		message.addTranslatedArg(Messages.PROPERTY_PATH_ARG, getPropertyPath());
		action.getMessages().addMessage(message);
	}
	
	
	private Set<String> getExistingFileIds(LoomServletRequest request) {
		Map<String, ParameterValue> params = RequestUtils.getParametersWithPrefix(request, UPLOADED_FILES_PREFIX);
		HashSet<String> result = new HashSet<String>();
		for (ParameterValue param : params.values()) {
			result.add(param.getString());
		}
		return result;
	}

	public void setFileManager(FileManager fileController) {
		this.fileManager = fileController;
	}
	
	public void setTransactionalService(ExtendedEntityManager service) {
		retrieveEntityInterceptor.setTransactionalService(service);
	}

	public void setRetrieveEntityInterceptor(RetrieveEntityInterceptor retrieveEntityInterceptor) {
		this.retrieveEntityInterceptor = retrieveEntityInterceptor;
	}

	public void setMaxFileSize(Long maxFileSize) {
		this.maxFileSize = maxFileSize;
	}

	public void setFilenameMaskPattern(Pattern filenameMaskPattern) {
		this.filenameMaskPattern = filenameMaskPattern;
	}

	public Long getMaxFileSize() {
		return maxFileSize;
	}

	public Pattern getFilenameMaskPattern() {
		return filenameMaskPattern;
	}
	
}
