/*
 * Copyright 2002-2006 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.loom.addons.autocompleter;

import javax.persistence.NoResultException;

import org.loom.binding.PropertyWrapper;
import org.loom.binding.PropertyWrapperFactory;
import org.loom.converter.AbstractConverter;
import org.loom.exception.RequiredAttributeException;
import org.loom.persistence.ExtendedEntityManager;
import org.loom.util.ClassUtils;

/**
 * Common code for single and multiple-valued autocompleted converters
 * @author icoloma
 */
public abstract class AbstractAutocompletedConverter extends AbstractConverter {

	/** the query to retrieve the instance from the database */
	private String query;
	
	/** the name of the persistent class to use */
	private Class<?> persistentClass;
	
	/** the name of the autocomplete property to use, default "name" */
	private String propertyName;
	
	private ExtendedEntityManager transactionalService;
	
	/** retrieve/set the value of the autocompleted property (propertyName) inside the persistentClass */
	private PropertyWrapper propertyWrapper;
	
	protected AbstractAutocompletedConverter() {
		// we don't know in advance the type of the returned object
		super(Object.class);
	}
	
	public void validate() {
		if (transactionalService == null) {
			throw new RequiredAttributeException("basicDao");
		}
		if (query == null) {
			query = "select o from " + persistentClass.getName() + " o where o." + propertyName + "=?";
		}
		propertyWrapper = new PropertyWrapperFactory().createInstance(persistentClass, propertyName);
	}
	

	/**
	 * @param propertyValue the value of the autocompleted property
	 * @return the persistent instance found with the provided value.
	 * @throws NoResultException if no entity with that value could be found
	 */
	protected Object retrieveEntity(String propertyValue) throws NoResultException {
		return transactionalService.findSingle(query, propertyValue);
	}

	public String getQuery() {
		return query;
	}

	public void setQuery(String query) {
		this.query = query;
	}

	public Class<?> getPersistentClass() {
		return persistentClass;
	}

	public void setPersistentClass(Class<?> persistentClass) {
		if (ClassUtils.isAbstract(persistentClass)) {
			throw new IllegalArgumentException(persistentClass.getName() + " cannot be used for @Autocompleted because it is abstract");
		}
		this.persistentClass = persistentClass;
	}

	public String getPropertyName() {
		return propertyName;
	}


	public void setPropertyName(String propertyName) {
		this.propertyName = propertyName;
	}

	public PropertyWrapper getPropertyWrapper() {
		return propertyWrapper;
	}

	public void setTransactionalService(ExtendedEntityManager entityManager) {
		this.transactionalService = entityManager;
	}
	
}
