/*
 * Copyright 2002-2006 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.loom.addons.autocompleter;

import java.lang.annotation.Documented;
import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

/**
 * Retrieve the annotated object from the database, according to an autocompleted 
 * property (usually, its name).
 * For example, if you prepare an autocompleter like this:
 * 
 * <pre>
 * <autocompleter name="customer.name"/>
 * 
 * \@Autocompleted(query="select Customer from Customer c where c.name=?", property="name")
 * private Customer customer;
 * </pre>
 * 
 * The customer.name request parameter will be intercepted and used to populate the customer property.
 * If no customer.name request param is found, the interceptor will do nothing.
 * 
 * @author Ignacio Coloma
 */
@Retention(RetentionPolicy.RUNTIME)
@Target({ElementType.FIELD, ElementType.METHOD})
@Documented
public @interface Autocompleted {

    /** if provided, applies to the property path instead of the annotated property */
    String propertyPath() default "";

    /** the list of events in which this validation takes place, empty for all. Accepts wildcards. */
    String[] on() default {};
    
    /** the list of events in which this validation does NOT place, empty for none. Accepts wildcards. */
    String[] except() default {};
    
    /** the name of the autocompleted property, default "name" */
    String propertyName() default "name";
    
    /** 
     * the query that will be launched to the database to retrieve the autocompleted entity.
     * If empty, a reasonable default will be assembled:
     * "select c from {object-class} c where c.{property-name}={property-value}
     */ 
    String query() default "";
    
    /** the list of tokens that separate each value,empty for single value. See Ajax.Autocompleter.tokens */
    String separatorTokens() default "";
    
}
