/**
 * A Tree implementation
 * Events: this implementation fires the following events:
 * tree:asyncOpen - Fired when a node triggered a successful request for its child nodes.
 * tree:selectNode - Fired when a node has been selected.
 *
 * AJAX Support
 * options {
 *   loader : function(id) {
 *       returns Arrays of { 
 *           id : String. Unique identified of the node. Must be also CSS unique. 
 *           title : String. Node title. could have simple markup, but no block tags.
 *           hasChildren : Boolean. Optional. Has children.
 *           children : Array of JSON nodes.
 *           class: String. Optional. Additional classes for node.
 *           unselectable: true | false. Default false.
 *         }
 *     }
 *     
 *   or
 *    
 *   loader : 'Ajax url' ||      Base url to load the json data from.
              user function,    Function that returns the list of nodes to apply
 *   loaderType : 'JSON' | 'HTML'. The HTML content should be <ul><li>...</li>...</ul>
 *
 *
 * The HTML generated will look like this:
 *  <ul class="tree">
 *    <li>
 *      <div class="item">
 *         <span class="spacer">
 *            <img src ... />
 *            ...
 *         </span>
 *         <img class="elbow ..." .../>
 *         <img class="icon ..." .../>
 *         <a class="anchor" ...><span class="title">...</span></a>
 *         <ul>
 *           ...
 *         </ul>
 *    <li>
 *  <ul>
 *
 * @author icoloma, rgrocha
 */
'loom.ui'.namespace();

loom.ui.Tree = Class.create({
    /**
     * Builds a new tree using the element as the source list.
     * @param {String or HTMLElement} element 
     * @param {Object or Hash} options Default options.
     * @constructor
     */
    initialize: function(element, options) {
            this.root = this.element = $(element);
            this.element.addClassName('tree');
        
        this.options = Object.extend({
                duration: 0.2, // duration of scriptaculous effects
                collapseAll: false, // true to collapse all tree nodes on init
                collapse: Element.hide, // the method used to collapse tree nodes
                expand: Element.show, // the method used to expand tree nodes
                onclick : Prototype.emptyFunction,
                spacer: null, // location of s.gif, if unspecified will be calculated 
                loaderType : 'JSON', // | 'HTML'
                loaderParamName : 'node', // name of the param name used for Ajax requests
                queue: 'end', // for script.aculo.us to wait when the user clicks too fast
                createLink: this.defaultCreateLink, // function that creates the link for each li item
                selectedNode: false // array with the selected node path to select after loading the tree
//              loader = 'http://ajaxurl' || User function
//              root: { 
//                    id: '',     // id of the created root node 
//                    class: '',  // CSS class of the created root node
//                    title: '' 
//              }
            }, options || {});
        
        // this is the less ugly way of calculating s.gif location. Note that we cannot use the script location,
        // as we don't know its name
        this.spacer = new Image();
        this.spacer.src = this.options.spacer || (function() {
			$$('body')[0].insert('<div id="treespacer__"> </div>');
			var s = $('treespacer__');
			var result = s.getStyle('background-image');
			s.remove();
			return /^url\(['" ]*([^'" ]+)['" ]*\)$/.exec(result)[1];
		})();
        
        // dynamic node creation
        if (options.loader)
            this.loader = typeof this.options.loader === 'string'? this.ajaxLoaderProcessor : this.overridenLoaderProcessor;
        this.onclick = this.options.onclick;
        
        // attach root node
        if (this.options.root) {
            
            this.element.insert({
                bottom : ('<ul class="tree tree-root"><li id="#{id}" class="#{class}">' +
                    '<div class="item"><img class="icon #{class}" src="#{spacer}"/><a class="anchor"><span class="title">#{title}</span></a></div>' +
                    '</li></ul>').interpolate({
	                    'id' : this.options.root.id || '',
	                    'class' : 'root folder ' + (this.options.root['class'] || ''),
	                    spacer: this.options.spacer,
	                    title :  this.options.root.title || ''
                    })
            });
            this.root = this.element.down('ul.tree-root');
            var rootLI = this.root.down('li');
            rootLI.spacer = '';
            this.options.selectedNode = this.options.selectedNode || this.options.root.id;
        }
        
        this.root.observe('click', this.toggle.bindAsEventListener(this));
        this.attachHTML(this.root);
        this.options.selectedNode && this.select(this.options.selectedNode);
    },
   
   /**
    * onClick event processor.
    * Toggle a tree item between selected and deselectes states, expanding
    * (and loading if necessary) or contracting its subtree, if any.
    * Also calls the user defined callback function. 
    * @private
    */
    toggle: function(evt) {
        var target = evt.target;
        var div = evt.findElement('div');
        var li = evt.findElement('li');
        var a = evt.findElement('a');
        var select = false;
        if (li) {
            if (target.hasClassName('folder')) {
                evt.stop();
                this.toggleNode(li, div);
            } else if (a) {
                if (!li.hasClassName('unselectable')) {
                	if (a.hasClassName('anchor')) {
                		evt.stop();
                		select = this.onclick(this, li.id);
                	} else {
                		return true;
                	}
                }
            }
            select && this.toggleSelect(li, div);
        }
    },
    
    /**
     * Removes all selected styles of the current item, if any.
     */
    unselectNode : function() {
        this.root.select('div.item.selected').invoke('removeClassName', 'selected');
    },
    
    /**
     * Turns on all selected styles on a given node, unselecting previous node before.
     * @param {HTMLElement} li Node to select
     * @param {HTMLElement} div Optional div.item inside the given li param (used to speedup the method)
     * @private
     */
    toggleSelect : function (li, div) {
        div = div || li.down('div.item');
        this.unselectNode();
        div.addClassName('selected');
        return li;
    },
        
    /**
     * Expands a folder node, showing its subtree.
     * If the subtree is not already loaded (and there is a defined loaded method)
     * loads and initialize it.
     * @param {HTMLElement} li Node to select
     * @param {HTMLElement} div Optional div.item inside the given li param (used to speedup the method) 
     * @param {Fucntion} onLoad Optional function only called if the subtree is loaded and after its initialized
     * @private
     */
    toggleNode : function(li, div, onLoad, forceExpand) {
        if (li.hasClassName('folder')) {
            div = div || li.down('div.item');
            
            var children = li.down('ul');
            
            // check for nodes to load
            if (!li.hasClassName('expanded') && !children) {
                if (this.loader) {
                    div.down('img.icon').addClassName('loading');
                    this.loader(li, onLoad);
                }
            } else {
            	forceExpand || !li.hasClassName('expanded')? this.expand(li) : this.collapse(li);
            }
        }
        return li;
    },
    
    /**
     * @private
     */
    removeLoadingAndExpand : function(li) {
        li.down('img.icon').removeClassName('loading');
        this.expand(li);
    },  
    
    /**
     * @private expand the specified node
     */
    expand : function(li) {
    	// display the children nodes
    	var children = li.down('ul');
    	children && this.options.expand(children, this.options);
    	
    	// add the class name
    	li.addClassName('expanded');
    	li.down('img.elbow').addClassName('expanded');
    	li.down('img.icon').addClassName('expanded');
    },
    
    /**
     * @private collapse the specified node
     */
    collapse : function(li) {
    	// display the children nodes
    	var children = li.down('ul');
    	children && this.options.collapse(children, this.options);
    	
    	// add the class name
    	li.removeClassName('expanded');
    	li.down('img.elbow').removeClassName('expanded');
    	li.down('img.icon').removeClassName('expanded');
    },
        
    /**
     * @private
     */
    overridenLoaderProcessor : function(li, onLoad) {
        var objList = this.options.loader(li.id);
        var children = this.attachJSON(li, objList.nodes);
        onLoad && onLoad(li, children);
        this.removeLoadingAndExpand(li, children);
    },
    
    /**
     * @private
     */
    ajaxLoaderProcessor : function(li, onLoad) {
        var params = {}
        params[this.options.loaderParamName] = li.id;
        new Ajax.Request(this.options.loader, {
                method : 'get',
                parameters : params,
                onSuccess: this.asyncOpen.bind(this, li, onLoad),
                onFailure: this.removeLoadingAndExpand.bind(this, li)
            });
    },
    
    /**
     * @private
     */
    asyncOpen : function(li, onLoad, transport) {
        var children;
        if (this.options.loaderType == 'JSON') {
            var objList = transport.responseText.evalJSON();
            children = this.attachJSON(li, objList.nodes);
        } else {
            var tmp = new Element("div");
            tmp.update(transport.responseText);

            tmp.select('ul').invoke('setStyle', {
                    overflow : 'visible',
                    display: 'none'
                });
            children = tmp.down('ul');
            li.appendChild(children);
            this.attachHTML(li);
            
            //temp.remove();
        }
        this.root.fire('tree:asyncOpen', li);
        onLoad && onLoad(li, children);
        this.removeLoadingAndExpand(li, children);
    },
        
    /**
     * Process the JSON object tree and creates the HTML list and attach it to the parent li.
     * @private
     */
    attachJSON : function (li, nodes) {
        var parentSpacer = li.spacer;
        if (nodes && nodes.length > 0) {
            var children = new Element('ul', {style : 'overflow: visible; display: none'});
            
            for (var i = 0; i < nodes.length; i++) {
                var node = nodes[i];
                
                var isLast = (i == nodes.length -1);
                var hasChildren = node.hasChildren || (node.children && node.children.length > 0);
                var isExpanded = false;
                
                var clazz = [];
                hasChildren && clazz.push('folder');
                isExpanded && clazz.push('expanded');
                node.unselectable && clazz.push('unselectable');
                
                var childLi = new Element('li', {
                        'id' : node.id,
                        'class' : clazz.join(' ')
                    });
                childLi.spacer = parentSpacer + '<img class="space' + (isLast? ' last':' line') + '" src="' + this.spacer.src + '"/>';

                this.createItem(childLi, node.title, isLast, hasChildren, isExpanded, li.spacer, node['class']);

                if (node.children && node.children.length > 0) {
                    childLi.addClassName('folder');
                    this.attachJSON(childLi, node.children);     
                }
                
                children.appendChild(childLi);
            }
            $(children.lastChild).addClassName('last');
            li.appendChild(children); 
            
            return children;
        }
    },
    
    /**
     * @private
     */
    attachHTML : function (element) {       
        element.cleanWhitespace();
        element.select('li:last-child').invoke('addClassName', 'last');
        element.select('li').each(function(li) {
                var isLast = li.hasClassName('last');
                var hasChildren = false;
                var isExpanded = false;
                
                var parentUL = li.up('li');
                var parentSpacer = (parentUL && parentUL.spacer)? parentUL.spacer : '';

                var childUL = li.down('ul');
                if (childUL || li.hasClassName('folder')) {
                    
                    hasChildren = true;
                    
                    if (childUL && childUL.visible()) {
                        isExpanded = true;
                    }
                    
                    li.spacer = parentSpacer + '<img class="space' + (isLast? ' last':' line') + '" src="' + this.spacer.src + '"/>';
                }
                                
                isLast && li.addClassName('last');
                hasChildren && li.addClassName('folder');
                isExpanded && li.addClassName('expanded');

                this.createItem(li, null, isLast, hasChildren, isExpanded, parentSpacer, li.readAttribute('itemClass'));
            }.bind(this));
    },
    
    /**
     * Collects te elements direct descendands text content, striping all tags from them.
     * Stops collecting text when it encounters an UL node.
     * @param {HTMLElement} element LI element
     * @returns A new span element containing the collected text. 
     * @private
     */
    getElementContent : function (element) {
        var title = new Element('span', {'class' : 'title'});
        while (element.firstChild && element.firstChild.nodeName != 'UL') {
            title.appendChild(element.firstChild);
        }
        title.innerHTML = title.innerHTML.stripScripts().stripTags();
        return title;
    },
        
    /**
     * @private
     */
    createItem : function(li, title, isLast, hasChildren, isExpanded, parentSpacer, itemClass) {
        var itemDiv = new Element('div', {'class' : 'item ' + (itemClass || '') });
        
        itemDiv.insert('<span class="spacer">#{parentSpacer}</span><img class="elbow#{elbow_class}" src="#{spacer_src}"/><img class="icon #{icon_class}" src="#{spacer_src}"/>'.
            interpolate({
                parentSpacer : parentSpacer,
                elbow_class : (hasChildren?' folder':'') + (isExpanded?' expanded':'') + (isLast?' last':''),
                spacer_src : this.spacer.src,
                icon_class : (itemClass || '') + (hasChildren?' folder':' leaf') + (isExpanded?' expanded':'')
            }));
        
        if (title) {
            title = new Element('span', {'class' : 'title'}).insert(title);
        } else {
            title = this.getElementContent(li);
        }

        li.insert({'top': itemDiv});

        itemDiv.insert(title.wrap(this.options.createLink(li)));
        
        return itemDiv;
    },
    
    /**
     * Create a link for the specified li
     */
    defaultCreateLink: function(li) {
        return new Element('a', {'href' : '#', 'class' : 'anchor'});
    },
    
    /**
     * @private
     */
    recursiveExpand : function(idpath, index) {
        // recorremos de manera no recursiva hasta que nos encontramos un nodo que no tenemos cargado
        var lastli;
        var li;
        do {
            lastli = li;
            li = this.root.select('#' + idpath[index++]).first();
            // make sure that the initialized nodes in the path are expanded
            if (li && !li.hasClassName('expanded')) {
            	//expand  only if the node has children, else use toggleNode
            	li.down('ul') && this.expand(li);
            } 
        } while (li && (index < idpath.length));
        if (li) {
            // This is the last node in the path so expand and select it.
            this.toggleNode(li, null, null, true);
            this.toggleSelect(li);
            this.root.fire('tree:selectNode', li);
        } else if (lastli) { // if the parent node exists at least
            this.toggleNode(lastli, null, function () {
                    this.recursiveExpand(idpath, index - 1);
                }.bind(this), true);
        }
    },
    
    /**
     * Selects a node and expand its subtree, if any.
     * @param {Objects} the list of node ids to load if necessary,
     * eg select(3, 15, 89). At least the first node must be already loaded.
     * It will also accept an array of item ids  
     */
    select : function() {
		this.recursiveExpand(Object.isArray(arguments[0])? arguments[0] : arguments, 0);
    }
    

});