/*
 * Copyright 2002-2006 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.loom.addons.autocompleter;

import java.io.IOException;
import java.util.Iterator;
import java.util.List;

import javax.servlet.jsp.JspException;

import org.loom.binding.PropertyWrapper;
import org.loom.converter.Converter;
import org.loom.tags.core.InputTextTag;
import org.loom.url.ParameterContainer;
import org.loom.url.UrlBuilder;
import org.loom.util.HtmlSanitizer;

/**
 * Prints an input field and a div for use with script.aculo.us Autocompleter.
 * The tag can be used specifying an action and event (and possibly nested &lt;l:param> tags),
 * or a list of options:
 * 
 * <pre>
 * &lt;a:autocompleter action="MyAction" event="myevent" name="mortgage.name">
 *    &lt;l:param name="param1" value="value1">
 * &lt;/a:autocompleter>
 * </pre>
 * 
 * @author icoloma
 */
public class AutocompleterTag extends InputTextTag implements ParameterContainer {

	/** the URL where an Autocompleter tag points to */
	static String AUTOCOMPLETER_URL = "autocompleter-url";

	/** the tokens used by an autocompleter tag to separate entries */
	static String AUTOCOMPLETER_SEPARATOR_TOKENS = "autocompleter-separator-tokens";


	/** if ajax, the default URL it points to */
	private UrlBuilder url = new UrlBuilder();
	
	/** 
	 * if not null, the array of autocompletion options that will be used with Autocompleter.Local.
	 * It can be either a list of Strings (which would be used as is) or, if an {@link Autocompleted}
	 * annotation is used, instances with a property that corresponds to its propertyName attribute.    
	 */
	private List<?> options;
	
	public AutocompleterTag() {
		setCssClass("autocompleter");
	}
	
	@Override
	public void doTagImpl() throws JspException, IOException {
		
		if (!url.isEmpty()) {
			setExtendedAttribute(AUTOCOMPLETER_URL, url.getURL(request));
		} else if (options != null) {
			identify();
			Converter c = getForm().getTargetAction().getConverter(getNormalizedName());
			if (c != null && c instanceof MultipleAutocompletedConverter) {
				MultipleAutocompletedConverter converter = (MultipleAutocompletedConverter) c;
				// autocalculate the tokens value if @Autocompleted was used
				setExtendedAttribute(AUTOCOMPLETER_SEPARATOR_TOKENS, ((MultipleAutocompletedConverter)converter).getSeparatorTokens());
			}
			printAutocompleterOptions(c);
		} else {
			throw new IllegalArgumentException("Either an options list or an action/event pair must be provided");
		}
		
		super.doTagImpl();

	}

	/**
	 * Prints the list of possible options
	 */
	protected void printAutocompleterOptions(Converter converter) throws IOException {
		StringBuilder arrayAsString = new StringBuilder();
		arrayAsString.append("[ ");
		
		PropertyWrapper property = null;
		if (converter != null && converter instanceof AbstractAutocompletedConverter) {
			property = ((AbstractAutocompletedConverter) converter).getPropertyWrapper();
		}
		for (Iterator<?> i = this.options.iterator(); i.hasNext(); ) {
			Object value = i.next();
			String sv;
			if (property != null) {
				value = property.get(value, false);
			} 
			sv = value == null? null : value.toString();
			arrayAsString.append('"').append(HtmlSanitizer.sanitize(sv)).append('"');
			if (i.hasNext())
				arrayAsString.append(", ");
		}
		arrayAsString.append(" ]");
		
		// maybe some day we get to use HTML 5 datalist, but right now it's impossible
		out.startJavascript();
		out.print("var ").print(getId()).print("_options = ").print(arrayAsString).print(";\n");
		out.endJavascript();
	}

	/**
	 * Set the target URL of this autocompleter (alternative to action)
	 */
	public void setUrl(String url) {
		this.url.setRawUrl(url);
	}

	public AutocompleterTag addParameter(String name, Object value) {
		url.addParameter(name, value);
		return this;
	}
	
	/**
	 * Set the target action of this autocompleter (alternative to url)
	 */
	public void setAction(String action) {
		url.setAction(action);
	}
	
	public void setEvent(String event) {
		url.setEvent(event);
	}

	public void setOptions(List<?> options) {
		this.options = options;
	}
	
}
