/*
 * Copyright 2002-2006 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.loom.addons.autocompleter;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;

import javax.persistence.NoResultException;

import org.apache.commons.lang.StringUtils;
import org.loom.i18n.Messages;
import org.loom.i18n.MessagesRepository;
import org.loom.servlet.params.ParameterValue;
import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

/**
 * Convert multiple autocompleted values ("customer1, customer2, customer3")
 * @author icoloma
 */
@Component
@Scope(BeanDefinition.SCOPE_PROTOTYPE)
public class MultipleAutocompletedConverter extends AbstractAutocompletedConverter {

	/** the list of tokens to use tokenized autocompletion */
	private String separatorTokens;

	@Override
	@SuppressWarnings("unchecked")
	public Object getAsObject(ParameterValue paramValue, Messages messages, MessagesRepository repository) {
		int i = 0;
		Collection result = createCollection();
		List<String> wrongValues = new ArrayList<String>();
		String sv = paramValue.getString();
		
		if (sv != null && sv.length() > 0) {
			for (String v : StringUtils.split(sv, separatorTokens)) {
				try {
					result.add(retrieveEntity(v));
				} catch (NoResultException e) {
					wrongValues.add(v);
				}
				i++;
			}
		}
		
		if (wrongValues.isEmpty()) {
			return result;
		}
		messages.addConversionErrorMessage("persistence.multipleAutocompletedEntityNotFound", paramValue, this).addArg("wrongValues", wrongValues);
		return null;
	}

	/**
	 * Create a collection holder. Right now, only Set and List are supported
	 */
	private Collection<?> createCollection() {
		return Set.class.isAssignableFrom(getConvertedClass())? new LinkedHashSet<Object>() : new ArrayList<Object>();
	}

	@SuppressWarnings("unchecked")
	public String getAsText(Object value, MessagesRepository repository) {
		if (value == null) {
			return null;
		}
		StringBuilder buffer = new StringBuilder();
		for (Iterator<Object>i = ((Iterable)value).iterator(); i.hasNext(); ) {
			buffer.append(getPropertyWrapper().get(i.next(), false));
			if (i.hasNext()) {
				buffer.append(separatorTokens.charAt(0));
			}
		}
		return buffer.toString();
	}

	public void setSeparatorTokens(String tokenSeparators) {
		this.separatorTokens = tokenSeparators;
	}

	public String getSeparatorTokens() {
		return separatorTokens;
	}

	

}
