package org.loom.addons.captcha;

import java.util.UUID;

import javax.servlet.http.HttpSession;

import org.loom.addons.servlet.names.RequestParameterNames;
import org.loom.interceptor.ExecuteInterceptor;
import org.loom.interceptor.FinallyListener;
import org.loom.mapping.ParsedAction;
import org.loom.resolution.Resolution;
import org.loom.servlet.LoomServletRequest;
import org.loom.util.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import com.octo.captcha.service.CaptchaService;

/**
 * Validates the Captcha. If the captcha value is not correctly introduced by the user,
 * adds a validation error message 
 * 
 * @author icoloma
 *
 */
@Component
public class CaptchaInterceptor implements ExecuteInterceptor, FinallyListener {

	@Autowired
	private CaptchaService captchaService;
	
	public Resolution beforeExecute(ParsedAction action) {
		LoomServletRequest request = action.getRequest();
		
		// check if this captcha was validated previously
		String validatedCaptchaID = request.getParameter(RequestParameterNames.VALIDATED_CAPTCHA_ID);
		HttpSession session = request.getSession(false);
		if (validatedCaptchaID == null || session.getAttribute(validatedCaptchaID) == null) {
			String captcha = request.getParameter(RequestParameterNames.CAPTCHA);
			String message = null;
			if (StringUtils.isEmpty(captcha)) {
				message = "captcha.error.missing";
			} else if (session == null || !captchaService.validateResponseForID(session.getId(), captcha)) {
				message = "captcha.error.incorrect";
			}
			if (message != null) {
				action.getMessages().addMessage(RequestParameterNames.CAPTCHA, message);
			} else {
				String validatedID = UUID.randomUUID().toString();
				request.addParameter(RequestParameterNames.VALIDATED_CAPTCHA_ID, validatedID);
				session.setAttribute(validatedID, Boolean.TRUE);
			}
		}
		return null;
	}

	public void doFinally(ParsedAction action, Exception e) {
		// if everything was OK, do the cleanup
		if (e == null && action.getMessages().isEmpty()) {
			LoomServletRequest request = action.getRequest();
			String validatedID = request.getParameter(RequestParameterNames.VALIDATED_CAPTCHA_ID);
			if (validatedID != null) {
				HttpSession session = request.getSession(false);
				session.removeAttribute(validatedID);
			}
		}
	}
	
	public void setCaptchaService(CaptchaService captchaService) {
		this.captchaService = captchaService;
	}
	
}
