/*
 * Copyright 2002-2006 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.loom.addons.hibernate;

import java.util.regex.Pattern;

import org.hibernate.validator.Email;
import org.hibernate.validator.Length;
import org.hibernate.validator.NotEmpty;
import org.loom.annotation.processor.AbstractAnnotationProcessor;
import org.loom.annotation.processor.metadata.AnnotationMetadata;
import org.loom.annotation.processor.metadata.FilteredAnnotationContainer;
import org.loom.annotation.validation.StringValidation;
import org.loom.mapping.ActionMapping;
import org.loom.validator.StringValidator;


/**
 * Processes Hibernate Validator annotations related to String validations: Pattern, Email, Length and NotEmpty
 * @author icoloma
 */
public class HibernateStringValidationAnnotationProcessor extends AbstractAnnotationProcessor {

	@Override
	@SuppressWarnings("unchecked")
	public void process(AnnotationMetadata metadata, ActionMapping mapping) {

		for (FilteredAnnotationContainer property : metadata.getAnnotatedProperties(org.hibernate.validator.Pattern.class, Email.class, Length.class, NotEmpty.class)) {
		
			org.hibernate.validator.Pattern pattern = property.getAnnotation(org.hibernate.validator.Pattern.class);
			org.hibernate.validator.Email email = property.getAnnotation(org.hibernate.validator.Email.class);
			org.hibernate.validator.Length length = property.getAnnotation(org.hibernate.validator.Length.class);
			org.hibernate.validator.NotEmpty notEmpty = property.getAnnotation(org.hibernate.validator.NotEmpty.class);
			
			if (pattern != null || email != null || length != null || notEmpty != null) {
				StringValidator validator = new StringValidator();
				if (length != null) {
					if (length.max() != Integer.MAX_VALUE) {
						validator.setMaxLength(length.max());
					}
					if (length.min() != 0) {
						validator.setMinLength(length.min());
					}
				}
				if (email != null) {
					Pattern p = Pattern.compile(StringValidation.EMAIL_MASK);
					validator.setMaskPattern(p);
					validator.setOverridenMessageKey("loom.validation.emailFailed");
				}
				if (pattern != null) {
					Pattern p = Pattern.compile(pattern.regex(), pattern.flags());
					validator.setMaskPattern(p);
				}
				validator.setPropertyPath(property.getPropertyPath());
				mapping.addValidator(property.getEventFilter(), validator);
			}
		}
		
	}

}
