/*
 * Copyright 2002-2006 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.loom.addons.multiupload;

import static org.loom.addons.servlet.names.RequestParameterNames.UPLOADED_FILES_PREFIX;

import java.io.IOException;
import java.util.List;

import javax.servlet.jsp.JspException;

import org.apache.commons.lang.StringUtils;
import org.loom.i18n.FileSizeFormatter;
import org.loom.mapping.ParsedAction;
import org.loom.persistence.file.PersistentFile;
import org.loom.addons.servlet.names.HtmlExtendedAttributeNames;
import org.loom.tags.AbstractDecoratedTag;
import org.loom.tags.WriterWrapper;
import org.loom.tags.core.InputFileTag;



/**
 * This tag generates multiple input file components in the browser.
 * By default it uses javascript code to generate the tags, but it can be configured to render
 * a number of file components in the server for accessible interfaces. 
 * @author icoloma
 */
public class MultiUploadTag extends AbstractDecoratedTag {

	private String name;
	
	/** true to add accessible HTML (consumes more bandwidth), defaults true */
	private boolean accessible = true;
	
	/** maximum number of rows to display, default to null if javascript is on and 10 if not */
	private Integer maxRows;
	
	private FileSizeFormatter fileSizeFormatter = new FileSizeFormatter();
	
	protected Object getValueAsObject() {
		ParsedAction parsedAction = request.getParsedAction();
		if (parsedAction == null)
			return null;
		return parsedAction.getPropertyAsObject(name); 
	}
	
	@SuppressWarnings("unchecked")
	public void doTagImpl() throws JspException, IOException {
		
		fileSizeFormatter.setMessagesRepository(getMessagesRepository());
		request.getCurrentForm().setMultipart(true);
		if (maxRows == null && accessible) {
			maxRows = Integer.valueOf(10);
		}
		
		out.print("<div");
		out.printAttribute("class", "multiupload");
		printParentAttributes();
		out.print(">\n");
		
		// Render the existing file IDs
		List<PersistentFile> files = (List<PersistentFile>) getValueAsObject();
		int index = 0;
		if (files != null && files.size() > 0) {
			out.print("<ul>\n");
			for (PersistentFile file : files) {
				render(out, file, index++);
			}
			out.print("</ul>\n");
		}

		// render multiple input file fields
		// * this code must generate the same HTML done via javascript
		// * this code will only be displayed in the browser if javascript is off
		if (accessible) {
			out.print("<noscript>\n<ul>\n");
			InputFileTag file = new InputFileTag();
			file.setWriter(out);
			file.setForm(request.getCurrentForm());
			file.setRequest(request);
			for (int i = 0; i < maxRows; i++) {
				file.setName(name + '[' + (index + i) + ']');
				out.print("<li>");
				file.doTagImpl();
				out.print("</li>\n");
			}
			out.print("</ul>\n</noscript>\n");
		}
		out.print("</div>\n");
			
	}

	/**
	 * Render one (already uploaded) file
	 * @param file
	 * @throws IOException 
	 */
	protected void render(WriterWrapper<?> out, PersistentFile file, int index) throws IOException {
		String filename = file.getFilename();
		out.print("<li");
		String extension = StringUtils.substringAfterLast(filename, ".");
		if (extension !=  null) {
			out.printAttribute("class", "file-" + extension);
		}
		out.print(">\n");
		
		// add checkbox to include file
		out.print("<input type=\"checkbox\"");
		out.printAttribute("name", UPLOADED_FILES_PREFIX + name + '[' + index + ']');
		out.printAttribute("value", file.getId());
		out.printAttribute("class", "checkbox");
		out.print(" checked/>");
		
		out.print("<span>");
		out.print(filename);
		if (file.getFileSize() != null) {
			out.print(" (").print(fileSizeFormatter.format(file.getFileSize())).print(")");
		}
		out.print("</span>\n");
        out.print("</li>\n");	

	}

	public Integer getMaxRows() {
		return maxRows;
	}

	public void setMaxRows(Integer maxRows) {
		this.maxRows = maxRows;
	}

	public boolean isAccessible() {
		return accessible;
	}

	public void setAccessible(boolean accessible) {
		this.accessible = accessible;
	}

	public String getName() {
		return name;
	}

	public void setName(String name) {
		this.name = name;
		setExtendedAttribute(HtmlExtendedAttributeNames.MULTIUPLOAD_NAME, name);
	}

}
