/**
 * Simple modal window
 * This class is intended for uses where modalbox may seem overkill
 */
loom.ui.Modal = Class.create({
	
	initialize: function(options) {
		this.options = Object.extend({
			lockScreen: false, // true to lock the interface so the user has to interact with the modal
			overlayOpacity: .5, // the opacity of the modal overlay
			animate: true // true to use script.aculo.us effects, false to use show/hide
		}, options || {});
		this.onClick = this.onClick.bindAsEventListener(this);
		this.overlay = this.createIfNotExists('div', { id: 'modalOverlay', style: 'display: none' });
		this.dialog = this.createIfNotExists('div', { id: 'modalDialog', style: 'opacity: 0' });
	},
	
	createIfNotExists: function(element, options) {
		var e = $(options.id);
		if (!e) {
			e =  new Element(element, options);
			$(document.body).insert(e);
		}
		return e;
	},
	
	/**
	 * Display the modal window
	 * @param html the HTML (string) or Element to put inside the window container
	 */
	show: function(html) {
		this.update(html);
		this.overlay.observe('click', this.onClick);
		var afterFinish = function() { this.dialog.fire('modal:show'); }.bind(this);
		
		if (this.options.animate) {
			this.overlay.appear({ to: this.options.overlayOpacity });
			this.dialog.appear( { afterFinish: afterFinish });
		} else {
			this.overlay.show();
			this.dialog.show();
			afterFinish();
		}
	},
	
	/**
	 * Updates the modal dialog with the given content and modifies its position
	 */
	update: function(html) {
		// ie6 = absolute, other browsers = fixed
		var offsets = this.dialog.getStyle('position') == 'absolute'? document.viewport.getScrollOffsets() : { top: 0, left: 0 };
		this.dialog.update(html);
		this.dialog.setStyle({ 
			top: (offsets.top + ((document.viewport.getHeight() - this.dialog.getHeight()) / 2).floor()) + 'px', 
			left: (offsets.left + ((document.viewport.getWidth() - this.dialog.getWidth()) / 2).floor()) + 'px'
		});
	},
	
	hide: function(event) {
		var afterFinish = function() {
			this.overlay.stopObserving('click', this.onClick);
			this.dialog.fire('modal:hide');
		}.bind(this);
		
		if (this.options.animate) {
			this.dialog.fade();
			this.overlay.fade({ afterFinish: afterFinish });
		} else {
			this.dialog.hide();
			this.overlay.hide();
			afterFinish();
		}
	},
	
	/**
	 * Default onclick handler: close if the user clicks outside the modal
	 */
	onClick: function(event) {
		if (!event.findElement('#modalDialog')) { // if clicked outside the modal
			!this.options.lockScreen && this.hide(event); // close the modal
			event.stop();
		}
	}
	
});
