/*  
 * The Loom framework
 * (c) Extrema Sistemas de Informacion
 *
 * Distributed under the Apache License, Version 2.0
 */

/**
 * Form autosave.
 * This script launches an autosave event with the current form contents
 * including any possible uploaded file. 
 * When autosaving, the new page is directed to a hidden iframe so the browser 
 * does not get affected.
 * For details about how multiple file uploading is implemented, check 
 * UploadedFilesInterceptor.java
 * 
 * @author icoloma
 */
var Autosave = {
    
    // when creating multiple input fields, this field checks the count
    multipleInputFileCount: 0,
    
    /** 
     * Sends the form to the server
     * Adds a loom-autosave param and calls the specified event in the same action
     * this form points to. The output is directed to a hidden frame so the window does 
     * not change its contents.
     * When the response is complete, the onSuccess method will be called.
     * This call involves file upload, so other ways XHR and copying to other forms
     * are not available. 
     * @param {Form} form the form to save
     * @param {String} event the event to call in the server. Cannot be null
     */
    autosave: function(form) {
        if (!form.dirty) {
            return;
        }
        Errors.showStatusMessage('');
        form.dirty = false;
        
        // replace the event with the autosave event
        console.debug("Autosaving form with event " + form.autosaveEvent);
		var oldEvent = Commons.setOrReplaceHidden(form, 'event', form.autosaveEvent);
        
        // add fields to know in the server side the id corresponding to each file upload field
        form.getElements().each(function (element) {
        	if (element.type == 'file') {
            	element.dirty = false;
            	//check that this happens!;
           	}
        });

		// the response should clear the error flag, unless there is an error
        Autosave.error = true; 
        
        // change target to a hidden iframe, and add autosave hidden field        
        form.target = 'autosave-iframe';
        form.submit();
        
        // clean the form for normal submit
        form.target = '';
		if (oldEvent) {
			$(form['event']).value = oldEvent;
		} else {
			$(form['event']).remove();
		}
    },
    
    onLoad: function(event) {
       // evalScripts() not needed, the browser does it by itself.
       if (Autosave.error) {
           var win = (event.target || event.currentTarget).contentWindow;
       	   var messages = win.ErrorMessages;
       	   if (messages) { // if validation errors happened
   	   		  for (var propertyName in messages) {
   	   		      if ($(propertyName)) { // maybe it has been removed by the user _after_ sending the form
	   	   		  	Validator.setError(propertyName, messages[propertyName]);
	   	   		  } else {
	   	   		    console.debug("messages[" + propertyName + "]=" + messages[propertyName]);
	   	   		  }
   	   		  }
		      Errors.showStatusMessage('The form could not be saved because of validation errors');
       	   } else { // standard exception / http error
	       	   var message = win.document.getElementById('errorMessage');
		       Errors.showStatusMessage(message? message.innerHTML : 'The form could not be saved', { error: true });
	       }
       } 
    },
    
    /**
     * Replaces an input type="file" with a hidden field and a link to remove
     * @param {String} propertyname the name attribute of the form field
     * @param {String} storedKey the key of the stored file in the server
     */
    replaceInput: function(name, storedKey) {
        $$('input[type="file"]').each(function(input) {
	        // if the input field has not been changed since it was uploaded to the server
	        if (input && input.name == name && !input.dirty) {
	            var filename = input.value.match(/[^\\/:*?"<>|]+$/)[0];
	            input.insert({after: new Element('input', {type: 'hidden', name: 'loom-uploaded-' + name, value: storedKey})});
	            input.insert({after: '<span>' + filename + '</span>'});
            	input.blur();
	            input.remove();
	        }
        });
    },
    
    /**
     * Called when a "form.autosave input" fires an onChange event.
     * Marks the form as dirty, so the next autosave sends it to the server. 
     * @param {input|select|textarea} input
     */
    onchange: function(event) {
        var element = event.target;
        element.form.dirty = true;
        if (element.type == 'file') {
            element.dirty = true;
        }
    },

    /** adds a new input file to the form when the "add new" button is clicked */
    addMultipleInputFile: function(event) {
        var ul = event.target.previous('ul');
        var form = ul.up('form');
        
        // copy the template row
        ul.insert(new Element('li'));
        var li = ul.immediateDescendants().last();
        
        // calculate the input field name
        var id = Autosave.getNextInputFileName(form, ul.name);
        li.insert(new Element('input', { type: 'file', name: id, id: id }));
        
        li.insert('<a href="#" class="removeFile">' + Resources.jsRemoveFile + '</a>')
        Event.observe(li.down('a'), "click", Autosave.removeMultipleInputFile); 
        if (form.hasClassName('autosave')) {
        	Event.observe(li.down('input'), "change", Autosave.onchange); 
       	}
       	event.stop();
    },
    
    /** return the next input file name */
    // TODO: change to Commons.getNextAvailableFieldName
    getNextInputFileName: function(form, inputName) {
		for (var i = 0; ; i++) {
			var candidate = inputName + '[' + i + ']';
			if (!form[candidate]) {
				return candidate;
			}
		}
    },
	
	/** Removes a file */
    removeMultipleInputFile: function(event) {
        event.target.up("li").remove();
        event.stop();
    },
    
    /** to be able to convert a date from an iframe, where extended (jsCalendar) Date methods are not available */
    dateToString: function(millis) {
    	return new Date(millis).print(Resources.jsDateTimeFormat);
    }
    
}

document.observe("dom:loaded", function() {
    $$('form.autosave').each(function(form) {
        form = $(form);
        var iframeId = "autosave-iframe";
        // for details about the iframe src, see 
        // http://aspnetresources.com/blog/always_set_iframe_source.aspx
        // http://www.mail-archive.com/prototype-core@googlegroups.com/msg01326.html
        form.insert(new Element("iframe", { name : iframeId, id: iframeId, src: '//:', 
            style: "width:0px; height:0px; border: 0px"}));
        new PeriodicalExecuter(function() { Autosave.autosave(form); }, form.autosaveInterval);
        $(iframeId).observe('load', Autosave.onLoad);
        form.select('input', 'select', 'textarea').invoke('observe', 'change', Autosave.onchange);
    });
    $$('ul.multiple-file').each(function(ul) {
    	var a = ul.next('.addfile');
    	if (a) {
    		a.observe('click', Autosave.addMultipleInputFile);
   		}
        ul.select('a.removeFile').invoke('observe', "click", Autosave.removeMultipleInputFile); 
    });
});
 
