/*  
 * The Loom framework
 * (c) Extrema Sistemas de Informacion
 *
 * Distributed under the Apache License, Version 2.0
 */

/**
 * Tag suggest component.
 *
 * @author icoloma at extrema-sistemas.com
 */
var TagSuggest = {

	defaultOptions: {
		tags: null, // list of available tags (array of string)
		delay: 0, // delay before asking for possible tags
		url: null, // url to get the list of available tags
		urlParams: null, // params  for the url
		separator: ' ' // used to separate tags 
	},

	initialize: function(element, options) {
		var options = Object.extend(Object.extend({ }, TagSuggest.defaultOptions), options || {});
		element.suggest = options;
		element.observe('keyup', TagSuggest.onKeyUp);
		element.observe('keypress', TagSuggest.onKeyPress);
	},
	
	// Intercepts the <enter> key, and if there are suggestions pending, use the first one available
	onKeyPress: function(event) {
		var evtc = event.keyCode;
		var suggestions = $(event.target.id + '-suggest');
		if (Event.KEY_RETURN == evtc && suggestions.innerHTML != '') {
			TagSuggest.completeTag(event.target, suggestions.select('a').first().innerHTML);
	      	Event.stop(event);
      	} 
	},
	
	// Searches possible tag completions after a key has been introduced
	onKeyUp: function(event) {
		var element = event.target;
		var candidate = TagSuggest.extractCandidate(element);
		
		// replace the suggestions div contents
		TagSuggest.clear(element.id);
		if (candidate.value != '') {
			var contents = '';
			var tags = TagSuggest.findMatches(element, candidate.value);
			$A(tags).each(function(tag) { contents += '<span><a href="#">' + tag + '</a></span>'});
			var suggestions = $(element.id + '-suggest');
			suggestions.update(contents);
			suggestions.select('a').invoke('observe', 'click', TagSuggest.onClick);
		}
	},
	
	// Remove any tag suggestions and stop observing the onClick event	
	// inputId: the input field ID
	clear: function(inputId) {
		var suggestions = $(inputId + '-suggest');
		suggestions.select('a').invoke('stopObserving', 'click');
		suggestions.update('');
	},
	
	// A suggested tag is clicked, completes the data being introduced with the suggested tag
	onClick: function(event) {
		var input = $(event.target.up('div').id.gsub('-suggest', ''));
		input.focus();
		TagSuggest.completeTag(input, event.target.innerHTML); 
		event.stop();
	},
	
	// completes the tag being introduced with the tag selected by the user
	completeTag: function(input, tag) {
		var candidate = TagSuggest.extractCandidate(input);
		input.value = input.value.substring(0, candidate.start) + tag + input.value.substring(candidate.end, input.value.length);
		TagSuggest.clear(input.id);
	},
	
	// returns an object with the following attributes
	// start: index of the first character position
	// end: index of the last character
	// value: text that is extracted between start and end
	extractCandidate: function(element) {
		var value = element.value;
		var caretPos = Selection.getStart(element);
		for (var init = caretPos; init > 0 && (value.charAt(init - 1) != element.suggest.separator); init--)
			;
		for (var end = value.charAt(caretPos) == element.suggest.separator? caretPos : caretPos + 1; end < value.length && value.charAt(end) != element.suggest.separator; end++)
			;
		var candidate = value.substring(init,  end).strip().toLowerCase();
		return { start: init, end: end, value: candidate };
	},
	
	// Find the tags that match a tag that is being introduced by the user
	// If there is one exact match, return an empty array, else return an array of tags
	// element: the input field
	// candidate: the last tag tyhat is being introduced
	findMatches: function(element, candidate) {
		var tags = [];
		return $A(element.suggest.tags).any(TagSuggest.match.curry(tags, candidate))? [] : tags;
	},
	
	// Return true if the string being introduced is a match for a tag.
	// If the tag is a partial match, adds it to the tags array. If it's a full match, returns true.
	// tags: the array of tags that have matched
	// candidate: the string that is being introduced
	// tag: the tag that is being compared
	match: function(tags, candidate, tag) {
		if (candidate == tag) {
			return true;
		} else if (tag.startsWith(candidate)) {
			tags.push(tag);
		}
	}

}

