/*  
 * The Loom framework
 * (c) Extrema Sistemas de Informacion
 *
 * Distributed under the Apache License, Version 2.0
 */

/**
 * Additional methods for the Number object
 */ 
Object.extend(Number.prototype, {

  // returns the number of digits
  precision: function() {
    return Math.abs(this).toString().replace('.', '').length;
  },
  
  // returns the number of decimal digits
  scale: function() {
    var stringValue = this.toString();
    var pos = stringValue.indexOf('.');
    return pos == -1? 0 : stringValue.length - (pos + 1);
  },

  // format a number and return a localized String
  format: function() {
    return this.toString().gsub('\\.', Resources.decimalSeparator);
  }

});

// parse a text and return a number
Number.parse = function(text) {
	text = text.gsub('\\' + Resources.groupingSeparator, '').gsub('\\' + Resources.decimalSeparator, '.');
    return parseFloat(text);
};

// a ValidationException
function ValidationException(message) {
  this.message = message;
  this.name = "ValidationException";
}

var Validator = {

  // returns true if the component is empty
  isEmpty: function(component) {
    return component.value.strip() == '';
  },
  
  // assigns an error message to the component
  setMessage: function(component, message) {
    component = $(component);
    var id = component.id || component.name;
    var errorMessageEcho = $("error_" + id);
    if (message) {
      if (!component.originalAlt)
        component.originalAlt = component.alt;
      component.alt = message;
      if (!component.originalTitle)
        component.originalTitle = component.title;
      component.title = message;
      if (errorMessageEcho) {
        errorMessageEcho.innerHTML = message;
        errorMessageEcho.show();
      } else 
        Errors.addMessage(id, message);
    } else {
      if (component.originalAlt)
        component.alt = component.originalAlt;
      if (component.originalTitle)
        component.title = component.originalTitle;
      if (errorMessageEcho != null) 
        errorMessageEcho.hide();
      else 
        Errors.removeMessage(id);
    }
  },
  
  // returns true if the component has an error
  hasError: function(component) {
    return component.hasClassName("error");
  },
  
  // clears the error message of a component
  clearError: function(component) {
    Validator.setMessage(component, null);
    component.removeClassName("error");
  },
  
  // sets the error message of a component
  setError: function(component, errorKey) {
    component = $(component);
    var errorMessage = Errors.translateMessage(component, errorKey);
    Validator.setMessage(component, errorMessage);
   	component.addClassName("error");
  },
  
  // add a validation method  
  addValidationMethod: function(component, method) {
    if (!component.validationMethods)
      component.validationMethods = [];
    component.validationMethods.push(method.bind(component));
  },
  
  // returns the value of this component, as a number
  getNumberValue: function(component) {
    var value = component.value;
	if (value == null)
		return null;
    if (!Resources.numberFormat.test(value))
      return NaN;
    return Number.parse(value);
  },
  
  // returns the value of this component, as a date
  getDateValue: function(component) {
    return Date.parseDate(component.value, component.dateFormat);
  },
  	
  // validates a component
  validate: function(event) {
    try { 
      component = event.target;
      if (!component.visible())
        return;
      if (component.validationMethods)
        component.validationMethods.each(function(method) {
          method(component);
        });
      Validator.clearError(component);
    } catch (e) {
      if (e instanceof ValidationException)
        Validator.setError(component, e.message);
      else
        throw e;
    }
  },

  // validates a required property
  validateRequired: function(component) {
    if (Validator.isEmpty(component)) {
      throw new ValidationException(component.requiredMessage || "loom.error.validation.requiredFailed");
    }
  },

  // validates a string property
  validateString: function(component) {
    if (Validator.isEmpty(component)) 
      return;
    var value = component.value.strip();
    var m = component.stringMessage;
    if (component.minLength != null && value.length < component.minLength)
      throw new ValidationException(m || "loom.error.validation.minLengthFailed");
    else if (component.maskPattern != null && !component.maskPattern.test(value))
      throw new ValidationException(m || "loom.error.validation.maskFailed");
  },

  // validates a numeric component
  validateNumber: function(component) {
    if (Validator.isEmpty(component)) 
      return;
    var value = Validator.getNumberValue(component);
    var m = component.numberMessage;
    if (isNaN(value))
      throw new ValidationException(m || "loom.error.conversion.numberFailed");
    else if (component.minValue != null && (component.excludeMin && value <= component.minValue || !component.excludeMin && value < component.minValue))
      throw new ValidationException(m || "loom.error.validation.minValueFailed");
    else if (component.maxValue != null && (component.excludeMax && value >= component.maxValue || !component.excludeMax && value > component.maxValue))
      throw new ValidationException(m || "loom.error.validation.maxValueFailed");
    else if (component.scale != null && component.scale < value.scale())
      throw new ValidationException(m || "loom.error.validation.scaleFailed");
    else if (component.precision != null && component.precision < value.precision())
      throw new ValidationException(m || "loom.error.validation.precisionFailed");
  },

  // validates a date
  validateDate: function(component) {
    if (Validator.isEmpty(component)) 
      return;
    var value = Validator.getDateValue(component);
    var m = component.dateMessage;
    
    // workaround: calendar.js will not complain if the input has wrong format, so this is needed
    if (value.print(component.dateFormat) != component.value) {
      throw new ValidationException(m || "loom.error.conversion.dateFailed");
    }
    if (component.dateMinValue != null && (component.excludeMin && value <= component.dateMinValue || !component.excludeMin && value < component.dateMinValue))
      throw new ValidationException(m || "loom.error.validation.dateMinValueFailed");
    else if (component.dateMaxValue != null && (component.excludeMax && value >= component.dateMaxValue || !component.excludeMax && value > component.dateMaxValue))
      throw new ValidationException(m || "loom.error.validation.dateMaxValueFailed");
  },
  
  // enforces that the textarea does not accept more than maxlength characters
  enforceMaxLength: function(event) {
    var evtc = event.keyCode; 
    var component = Event.element(event);
  	if (component.value.length >= component.maxLength && Event.SPECIAL_KEYS.indexOf(evtc) == -1) {
        Event.stop(event);
    }
  },
  
  // bind validations to a single element
  bindValidationToElement: function(element) { 
	  
    element.observe("change", Validator.validate);
    if (element.hasClassName("required"))
      Validator.addValidationMethod(element, Validator.validateRequired);
    if (element.hasClassName("number"))
      Validator.addValidationMethod(element, Validator.validateNumber);
    if (element.hasClassName("string"))
      Validator.addValidationMethod(element, Validator.validateString);
    if (element.hasClassName("date") || element.hasClassName("dateTime")) {
      Validator.addValidationMethod(element, Validator.validateDate);
    // min and max are passed as millis
	  element.dateMinValue = new Date(parseInt(element.dateMinValue));
	  element.dateMaxValue = new Date(parseInt(element.dateMaxValue));
    }
       // textfield have a maxlength attribute, but textarea does not
       // this method enforces a textarea maxlength attribute
	if (element.tagName == 'TEXTAREA' && element.maxLength) { 
		element.observe('keypress', Validator.enforceMaxLength);
           // because paste etc are not included in the "keypress" listener:
		element.observe('change', function(event) {
               component = Event.element(event);
               component.value = component.value.substring(0, component.maxLength);
           });
	}
      
  },
  
  bindValidations: function() {

	  $$('input.validate', 'textarea.validate', 'select.validate').each(Validator.bindValidationToElement);
	  
	  $$('input.date', 'input.dateTime').each(function(element) {
	    element.dateFormat = element.hasClassName('date')? Resources.jsDateFormat : Resources.jsDateTimeFormat;
	    
	    // quick hack to print the maximum allowed Date
        if (element.dateMinValue) {
	       element.minValue = { date: element.dateMinValue.print(element.dateFormat) };
	    }
	    if (element.dateMaxValue) {
	       element.maxValue = { date: element.dateMaxValue.print(element.dateFormat) };
	    }
	    
	    Calendar.setup({
	        inputField : element.id,
	        ifFormat   : element.dateFormat,
	        showsTime  : element.hasClassName('dateTime'),
	        button     : ("btn-" + element.id).camelize(),
	        onUpdate   : function(cal) { Validator.validateDate(cal.params.inputField); }
	    });
	  });
	  
	  $$('input.autocomplete').each(function(element) {
		new Ajax.Autocompleter(element.id, element.id + '.choices', element.targetUrl, {
			frequency: element.frequency,
			minChars: element.minChars
		});
	  });
	  
	  // select the first error element
	  $$('input.error').any(function(element) {
	  	element.activate();
	  	return true;
	  });
  }
  
}
    
// keeps the list of errors of this form up-to-date
var Errors = {

  ParameterPattern: /(^|.|\r|\n)(\$\{(.*?)\})/,
  
  // returns the translated message
  translateMessage: function (component, messageKey) {
    // override if the user has specified a custom message
    messageKey = component.message || messageKey;
    var messageTemplate = Resources[messageKey];
    if (messageTemplate == null)
      return messageKey;
	//normalize property path
	var key = component.name.gsub(/\[\w+\]/, '');
    var params = {
      "propertyName": PropertyNames[key] || component.name,
      "validator": component,
      "value": component.value,
      "length": component.value.length
    }
    return messageTemplate.interpolate(params, Errors.ParameterPattern);
  },
  
  // adds an error message for a given component
  addMessage: function (componentId, message) {
    var errors = $("errors");
    if (errors == null) {
      console.error(message);
      return;
    }
    id = "errors_" + componentId;
    var item = $(id);
    if (item == null) {
      var ul = errors.firstDescendant();
	  if (!ul) {
	  	ul = errors.insert(new Element('ul'));
	    ul = errors.firstDescendant();
	  }
      item = document.createElement('li');
      item.id = id;
      ul.appendChild(item);
    }
    item.innerHTML = message;
    errors.show();
  },
  
  // removes the error message of a given component
  removeMessage: function (componentId) {
    var item = $("errors_" + componentId);
    if (item != null) {
      var ul = item.parentNode;
      item.remove();
      if (ul.immediateDescendants().length == 0)
        ul.parentNode.hide();
    }
  },
  
  // echo the status of the last operation
  // if there is a 'status' div in the page, update its contents; else, log in the firebug console
  // options {
  // error: true if this is an error message
  // }
  // rest of the options are passed as is to the message template
  showStatusMessage: function(message, options) {
    options = options || {};
    // translate client-side parameters such as 'browserLocalTime'
    message = new Template(message, Errors.ParameterPattern).evaluate(options);
    var status = $('status'); 
  	if (status) {
  		status[options.error? 'addClassName' : 'removeClassName']('error');
  		status.update(message);
  		status[!message || message.empty()? 'hide' : 'show']();
   	} else {
		console[options.error? 'error' : 'info'](message);
	}
  }
  
}

document.observe("dom:loaded", Validator.bindValidations);
