/*  
 * The Loom framework
 * (c) Extrema Sistemas de Informacion
 * Distributed under the Apache License, Version 2.0
 */
if ((typeof Prototype == 'undefined') || Prototype.Version < '1.6') {
    throw new Error('Loom requires Prototype 1.6.0 or greater');
}

/**
 * Original implementation by by kangax and henrah 
 * at http://thinkweb2.com/projects/prototype/namespacing-made-easy/
 */
String.prototype.namespace = function(separator) {
  this.split(separator || '.').inject(window, function(parent, child) {
    return parent[child] = parent[child] || { };
  })
}

var loom = {

	/** loom GET params prefix */
	PREFIX: '__', 
	
	/** local time offset, in hours (2 means UTC+0200) */
	timezone: new Date().getTimezoneOffset() / -60,
	
	i18n: {},
	
	/** true if this is a IE <7 */ 
	ie6: Prototype.Browser.IE && navigator.appVersion < "4.0 (compatible; MSIE 7",
  
	/**
	 * Original implementation by Carlos Reche:
	 * http://wiki.script.aculo.us/scriptaculous/show/Cookie 
	 */
	cookies: {
	
	  set: function(name, value, daysToExpire, path) {
	    var expire = '';
	    if (daysToExpire != undefined) {
	      var d = new Date();
	      d.setTime(d.getTime() + (86400000 * parseFloat(daysToExpire)));
	      expire = '; expires=' + d.toGMTString();
	    }
	    path = '; Path=' + (path || '/');
	    return (document.cookie = escape(name) + '=' + escape(value || '') + path + expire);
	  },
	  
	  get: function(name) {
	    var cookie = document.cookie.match(new RegExp('(^|;)\\s*' + escape(name) + '=([^;\\s]*)'));
	    return (cookie ? unescape(cookie[2]) : null);
	  },
	  
	  remove: function(name) {
	    var cookie = loom.cookies.get(name);
	    loom.cookies.set(name, '', -1);
	    return cookie;
	  },
	  
	  /**
	   * return true if the browser has cookies enabled
	   */
	  accept: function() {
	    if (typeof navigator.cookieEnabled == 'boolean') {
	      return navigator.cookieEnabled;
	    }
	    loom.cookies.set('_test', '1');
	    return (loom.cookies.remove('_test') === '1');
	  }
    
	},
	
	/**
	 * Return the full path of the provided js script without the script name, e.g.<b>
	 * loom.getScriptPath('prototype.js') returns 'http://prototype.org/js/script' if the page 
	 * includes http://prototype.org/js/script/prototype.js?something.
	 * If the script is not found, returns null.  
	 */
	getScriptPath: function(filename) {
	    var path = null;
	    $A(document.getElementsByTagName("script")).any( function(s) {
	       var src = s.getAttribute('src');
	       if (src) {
		       var pos = src.indexOf(filename);
	           return pos != -1 && (path = src.substring(0, pos));
           }
        });
        return path;
	}
	
	
}/** keycodes that are not text */
loom.ui = {

  /** substitute with "l:" or similar to use namespaced XHTML attributes instead */
  extendedPrefix: 'data-',

  event: {
  
	  SPECIAL_KEYS: [ 
	    Event.KEY_BACKSPACE, Event.KEY_TAB, Event.KEY_RETURN, Event.KEY_ESC, Event.KEY_LEFT, 
	    Event.KEY_UP, Event.KEY_RIGHT, Event.KEY_DOWN, Event.KEY_DELETE, Event.KEY_HOME, 
	    Event.KEY_END, Event.KEY_PAGEUP, Event.KEY_PAGEDOWN
	    ]
	    
	},
	
	/**
	 * Return the (translated) property name  of an element
	 * If there is a surrounding label or a label with the corresponding for attribute,
	 * return its text contents.
	 * Else return element.name   
	 */
	getPropertyName: function(element) {
	  var label = element.up('label');
	  if (!label && element.id) 
	    label = $$('label[for=#{id}]'.interpolate({id: element.id})).first();
	  if (label) // if there is a nested span tag, use it instead
	    label = label.down('span') || label;
	    
	  // if there is a label, return that
	  return label? (label.innerText || label.textContent).strip().gsub(':', '') : element.name;
	  
	},
	
	/**
	 * Creates a popup calendar bound to an input text field 
	 * @param {Element} element the input type=text" field to bind the calendar to
	 */
	createDatePicker: function(element) {
		element = $(element);
		if (!element.disabled) {
			element.setAttribute("autocomplete", "off");
			element.insert({after: '<a class="dateButton">open calendar</a>'});
			var button = element.next('.dateButton');
			Calendar.setup({
			  dateField      : element,
			  triggerElement : button,
			  dateFormat: loom.i18n.resources['loom.format.jsDate'],
	          selectHandler: Calendar.defaultSelectHandler.wrap(function(proceed, calendar) { // fire select event
				  proceed(calendar);
				  element.fire('calendar:change');
			  })
			});
			element.observe('calendar:change', function(e) {
			  element.validate && element.validate();
			});
		}
	},
  
  /**
   * Creates an Autocompleter.Local or Ajax.Autocompleter depending on the field attributes
   * @param options the list of options that will be propagated to the created autocompleter
   * @return the created instance
   */
  createAutocompleter: function(element, options) {
    element = $(element);
    var choicesDisplay = $('choices');
    if (choicesDisplay == null) {
      $(document.body).insert('<div id="choices" style="display:none"> </div>');
      choicesDisplay = $('choices');
    }
    
    var choices = window[element.id + '_options'];
    if (choices) { 
      return new Autocompleter.Local(element, choicesDisplay, choices, options);
    } else { 
      new Ajax.Autocompleter(element, choicesDisplay, element.getExtendedAttribute('autocompleter-url'), options);
    }
    
  }
	
}

Element.addMethods('form', {
   
   // search the next valid index for a multiple field 
   // template: the template to search with syntax 'my.nested.propertyName[${index}]'
   // return the next available field name
   getNextAvailableFieldName: function(form, template) {
   	  for (var i = 0; i < 100; i++) {
   	     var candidate = template.interpolate({ index: i }, /(^|.|\r|\n)(\$\{(.*?)\})/);
   	     if (!$(candidate)) {
   	       return candidate;
   	     }
   	  }
   	  throw new Error('Could not find next field name (max iterations reached).');
   }
   
});

Element.addMethods({

  /**
   * element: the element where to search for this attribute
   * name: the name of the attribute
   * defaultValue: if present, the value to return if no such attribute is found.
   * converter: the method to call to convert the string value to an object type.
   */
  getExtendedAttribute: function(element, name, defaultValue, converter) {
    return (converter || Prototype.K)(element.getAttribute(loom.ui.extendedPrefix + name) || defaultValue);
  },
  
  getExtendedAttributeAsBoolean: function(element, name, defaultValue) {
    return element.getExtendedAttribute(name, defaultValue, function(v) { return /^true$/.match(v); });
  },
  
  getAttributeAsDate: function(element, name) {
    var v = element.getAttribute(name);
    return v? loom.format.parseDate('%i', v) : null;
  }
  
});
/**
 * Confirmation before following a link or submitting a form.
 */
loom.ui.Confirmation = Class.create({

  initialize: function(element) {
    this.element = element;
    this.confirmationId = element.getExtendedAttribute('confirmation-id');
    this.message = element.getExtendedAttribute('confirmation-message', this.confirmationId);
  },
    
  // intercepts the click event over a link or button element
  onClick: function(event) {
    event.stop();
    this.openWindow();
  },
  
  // intercepts the click over a confirmation button (accept or cancel)
  onConfirmationClick: function(e) {
    var a = e.findElement('a');
    if (a != null) {
      e.stop();
      this.closeWindow();
      if (a.hasClassName('accept')) 
        this.onAccept(e);
    }
  },
  
  /**
   * Override to use a different window system
   * message: the window message
   * onAccept: the handler if the user clicks accept
   */
  openWindow: function() {
    if (!$('confirmationBuffer'))
      $(document.body).insert('<div id="confirmationBuffer" style="display:none"></div>');
  
    var html = ('#{message}' +
    '<div id="confirmation-buttons"><a href="#" class="accept">#{accept}</a><a href="#" class="cancel">#{cancel}</a>' +
    '</div>').interpolate({
      message: loom.i18n.resources[this.message + '.message'],
      accept: loom.i18n.resources[this.message + '.accept'],
      cancel: loom.i18n.resources[this.message + '.cancel']
    });
    var c = this;
    Modalbox.show(html, { 
      title: loom.i18n.resources[this.message + '.title'], 
      afterLoad: function() { 
		    $('confirmation-buttons').observe('click', c.onConfirmationClick.bindAsEventListener(c));
      }
    });
    
  },
  
  /**
   * Override to use a different window system
   */
  closeWindow: function() {
    Modalbox.hide();
  }
  
});

Object.extend(loom.ui, {

	LinkConfirmation: Class.create(loom.ui.Confirmation, {
	
    initialize: function($super, a) {
      $super(a);
      this.targetUrl = a.href + (a.href.include('?') ? '&' : '?') + loom.PREFIX + "confirmation-" + this.confirmationId + "=true";
    },
    
    onAccept: function(e) {
      window.location.href = this.targetUrl;
    }
    
	}),
	
	FormConfirmation: Class.create(loom.ui.Confirmation, {
	
    onAccept: function(e) {
      var button = this.element;
      var form = button.up('form');
      form.insert(new Element("input", { name: loom.PREFIX + "confirmation-" + this.confirmationId, value: 'true', type: 'hidden' }));

      this.closeWindow();
      if (button.name.startsWith(loom.PREFIX + 'event-')) { // overriden event
        element = new Element("input", { name: button.name, value: 'true', type: 'hidden' });
        element.identify();
        form.insert(element);
        this.submit(form);
        element.remove();
      } else {
        this.submit(form);
      }
    },
    
    // this method is here just for testcases
    submit: function(form) {
      form.submit();
    }
    
 	}),
 	
 	bindConfirmations: function(element) {
    if (Object.isArray(element)) {
      element.each(loom.ui.bindConfirmations);
      return;
    } 
    var c;
    if (element.tagName == 'A') {
      c = new loom.ui.LinkConfirmation(element);
    } else if (element.match('input[type=submit]')) {
      c = new loom.ui.FormConfirmation(element);
    } else  {
      throw new Error('Unable to handle element ' + element);
 	}
    element.observe('click', c.onClick.bindAsEventListener(c));
    }
	
});
/**
 * Based on previous work from 
 *
 * Alvaro Isorna:
 * http://snipplr.com/view.php?codeview&id=125
 *
 * Andreas Everhard:
 * http://jquery.andreaseberhard.de/pngFix/index.html
 *
 * And myself :)
 * http://icoloma.blogspot.com
 *
 * Migrated from jQuery and cleaned up big time.
 */
loom.ui.PNG = {
	
	template: 'progid:DXImageTransform.Microsoft.AlphaImageLoader(src="#{src}",sizingMethod="#{method}")',
	
	/**
	 * Fix a PNG background
	 */
	fixBackground: function(element) {
	
	  var src = element.getStyle('backgroundImage').match(/["'(]+(.*\.png)/i)[1];
	  
	  element.setStyle({
	    filter: loom.ui.PNG.template.interpolate({ src: src, method: element.getStyle('backgroundRepeat') == "no-repeat" ? "crop" : "scale"}),
	    backgroundImage: "none"
	  });
	  
	},
	
	/**
	 * Fix a PNG image
	 */
	fixImage: function(element) {
	      
	      var span = new Element('span', {
	      
	        id: element.id,
	        className: element.className,
	        title: element.title,
	        alt: element.alt
	        
	      });
	      span.setStyle({
	      
	        width: element.width, 
	        height: element.height,
	        filter: loom.ui.PNG.template.interpolate({ src: element.src, method: 'scale' }),
	        position: 'relative',
	        'white-space': 'pre-line',
	        display: 'inline-block',
	        background: 'transparent',
	        'float': element.style['float'],
	        cursor: element.up('a')? 'hand' : null,
	        border: element.style.border,
	        padding: element.style.padding,
	        margin: element.style.marginbgIMG
	      });
	      
	      element.replace(span);
	}
	
}

loom.ui.PNG.fix = function() {
  
    // fix the background image of any element with the fix-png css class
    $$('.fixpng').each(loom.ui.PNG.fixBackground);
    
    // fix png images
    $$('img[src$=".png"]').each(loom.ui.PNG.fixImage);
	
}

// disable for grade-A browsers
if (!loom.ie6) {
    $w('fix fixImage fixBackground').each(function(f) {
		loom.ui.PNG[f] = Prototype.K;
	});
}'loom.format'.namespace();

/**
 * Date parsing and formatting libraries
 * @author icoloma
 */
Object.extend(Date.prototype, {
	
  /**
   * Original implementation posted by Justin Palmer
   * http://alternateidea.com/blog/articles/2008/2/8/a-strftime-for-prototype
   * @param {String} format the format string as specified by PHP's strftime function 
   */
  format: function(format) {
    // replace token aliases prior to formatting
    var parser = new loom.format.DateParser(format);
    format = parser.format;

    var day = this.getDay(), month = this.getMonth();
    var hours = this.getHours(), minutes = this.getMinutes();
	  var timezone = this.getTimezoneOffset() / -60;
    function pad(num) { return num.toPaddedString(2); };

    return format.gsub(/\%([aAbBcdDHiImMpSwyYZz%tn])/, function(part) {
      switch(part[1]) {
        case 'a': return parser.shortDayNames[day]; break;
        case 'A': return parser.dayNames[day]; break;
        case 'b': return parser.shortMonthNames[month]; break;
        case 'B': return parser.monthNames[month]; break;
        case 'c': return this.toString(); break;
        case 'd': return pad(this.getDate()); break;
        case 'H': return pad(hours); break;
        case 'i': return (hours === 12 || hours === 0) ? 12 : (hours + 12) % 12; break;
        case 'I': return pad((hours === 12 || hours === 0) ? 12 : (hours + 12) % 12); break;
        case 'm': return pad(month + 1); break;
        case 'M': return pad(minutes); break;
        case 'P': return hours > 11 ? 'pm' : 'am'; break;
        case 'p': return hours > 11 ? 'PM' : 'AM'; break;
        case 'S': return pad(this.getSeconds()); break;
        case 'w': return day; break;
        case 'y': return pad(this.getFullYear() % 100); break;
        case 'Y': return this.getFullYear().toString(); break;
    		case 'z':
    		case 'Z': return timezone == 0? 'Z' : (timezone > 0? '+' : '-') + pad(timezone.abs()); break;
    		case '%': return '%'; break;
    		case 't': return '\t'; break;
    		case 'n': return '\n'; break;
      }
    }.bind(this));
  }
});

    // initialize token aliases
loom.format.aliases = $H({
  F: '%Y-%m-%d', 
  r: '%I:%M:%S %p', // time in a.m. and p.m. notation
  R: '%H:%M', // time in 24 hour notation 
  T: '%H:%M:%S', // current time
  D: '%m/%d/%y',
  i: '%Y-%m-%dT%H:%M:%S%z' // ISO 8601 date format (extension over the strftime format)
  //c: loom.i18n.resources['loom.format.jsDateTime'],
  //x: loom.i18n.resources['loom.format.jsDate'], // preferred date representation for the current locale without the time
  //X: loom.i18n.resources['loom.format.jsTime'] // preferred time representation for the current locale without the date                   
});

loom.format.DateParser = Class.create({

  /**
   * Create  new parser
   * @return {Parser} a parser instance 
   * @param {String} format
   */
  initialize: function(format) {
    
    // initialize this.dayNames, this.shortDayNames, this.monthNames, this.shortMonthNames
    $w('dayNames monthNames').each(function(n) {
      this[n] = $w(loom.i18n.resources['loom.format.' + n]);
      var n2 = ('short-' + n).camelize();
      var v2 = loom.i18n.resources['loom.format.' + n2];
      this[n2] = v2? $w(v2) : this[n].map(function(s) { return s.substring(0, 3)}); 
    }.bind(this));
    
    // replace token aliases 
    format = this.resolveAliases(format);
    this.allTokens = this.getAllTokens();
    this.format = format;
    this.regex = '^';
    this.tokens = $A([]);
    for (var i = 0; i < format.length; ) {
      var c = format.charAt(i++);
      if (c == '%') {
        var token = this.allTokens[format.charAt(i++)];
        if (!token)
          throw new SyntaxError("Unknown directive: %" + format.charAt(i-1));
        this.tokens.push(token);
        this.regex += '(' + token.ex + ')';
      } else {
        this.regex += c;
      }
    }
    this.regex = new RegExp(this.regex + '$', "i"), 
    
    loom.format.dateParsers[format] = this;
  },
  
  /**
   * Expected tokens, according to the expected format.
   * For each token, the token data must follow this structure:
   * { 
   * ex: the regular expression that defines this word
   * f the function that parses the input text, null for Prototype.K
   * p the date property that is set with this value, null to discard the value
   * ampm either one of 'am', 'pm' or null if the hour is in 24-hour format 
   * timezone the timezone, if any
   * }
   *
   */
  getAllTokens: function() { 
    var t = { 
    
      a: { // abbreviated weekday name according to the current locale
        ex: this.shortDayNames.join('|')
      },
      A: { // full weekday name according to the current locale
        ex: this.dayNames.join('|')
      },
      b: { // abbreviated month name according to the current locale
        ex: this.shortMonthNames.join('|'),
        f: function(text, parser) { return parser.shortMonthNames.indexOf(text) },
        p: 'month'
      },
      B: { // full month name according to the current locale
        ex: this.monthNames.join('|'),
      f: function(text, parser) { return parser.monthNames.indexOf(text) },
      p: 'month'
      },
      d: { // day of the month as a decimal number (range 01 to 31)
        ex: "\\d?\\d",
      p: 'day'
      },
      H: { // // hour as a decimal number using a 24-hour clock (range 00 to 23)
        ex: "\\d?\\d",
        p: 'hour'
      },
      I: { // hour as a decimal number using a 12-hour clock (range 01 to 12)
        ex: "\\d?\\d",
      //f: function(v) { return parseInt(v) - 1; },
        p: 'hour'
      },
      m: { // month as a decimal number (range 01 to 12)
    
        ex: '\\d?\\d',
      f: function(m) { return m - 1 },
      p: 'month'
      },
      M: { // minutes as a decimal number
        ex: '\\d?\\d',
      p: 'minute'
      },
      n: { // carriage return
        ex: '\\n'
      },
      p: { // // 'AM'/'PM' literal
        ex: 'am|pm',
        f: function(m) { return m.toLowerCase(); },
        p: 'ampm'
      },
      S: { // seconds as a decimal number. Milliseconds are optional, but only JodaTime uses them AFAIK. Do not use parenthesis here!
        ex: '\\d{1,2}|\\d{1,2}\\.\\d{3}',
      p: 'second'
      },
      t: { // tab
        ex: '\\t'
      },
      u: { // weekday as a decimal number [1,7], with 1 representing Monday
        ex: '\\d'
      },
      y: { // year as a decimal number without a century (range 00 to 99)
        ex: '\\d\\d',
      f: function(v) { v = parseInt(v); return v > 80? 1900 + v : 2000 + v},
      p: 'year'
      },
      Y: { // year as a decimal number including the century
        ex: '\\d{4}',
      p: 'year'
      },
      Z: { // timezone name or Z for UTC. Do not use parenthesis here!
        ex: 'Z|[+-]\\d{2}|[+-]\\d{4}|[+-]\\d{2}:\\d{2}',
      f: function(v) { 
        if (v == 'Z') // UTC
            return 0;
          var a = /^([+-]\d{2}):?(\d{2})?$/.exec(v); // hour and (optional) minutes
        return parseInt(a[1]) + (!a[2]? 0 : parseInt(a[2]) / 60); 
      },
      p: 'timezone'
      },
      '%': { // '%' character
        ex: '\\%'
      }
    
    };
    
    // format aliases
    t = Object.extend(t, {
      h: t.b, // abbreviated month name according to the current locale
      w: t.u, // day of the week as a decimal, Sunday being 0
      P: t.p, // am/pm as lowercase
      z: t.Z // timezone
    });
    
    return t;
  
  },

  resolveAliases: function(format) {
    loom.format.aliases.each(function(pair) {
        format = format.gsub('%' + pair.key, pair.value);
    });
    return format;
  },
  
  parse: function(text) {
    var v = this.regex.exec(text);
    if (!v)
    return NaN;
    
    // process all tokens, if t.p is not null
    var date = {};
    var parser = this;
    this.tokens.each(function(t, index) {
      if (t.p) {
        date[t.p] = (t.f || Prototype.K)(v[index + 1], parser);
      }
    });
  
      
    if (date.hour && date.ampm == 'pm')
      date.hour = parseInt(date.hour) + 12;
    
    var d = loom.format.createDate(date.year, date.month, date.day, date.hour, date.minute, date.second);
    
    // adjust timezone
    if (date.timezone != null)
      d = new Date(d.getTime() + (date.timezone - loom.timezone) * 3600000); 
  
    return d;
  }
  
});

/**
 * Handle decimal numbers format and parsing
 */ 
loom.format.NumberFormat = Class.create({

  initialize: function(number) {
    this.number = number;
  },

  // returns the number of digits
  precision: function() {
    return Math.abs(this.number).toString().replace('.', '').length;
  },
  
  // returns the number of decimal digits
  scale: function(number) {
    var stringValue = this.number.toString();
    var pos = stringValue.indexOf('.');
    return pos == -1? 0 : stringValue.length - (pos + 1);
  },
  
  // format a number and return a localized String
  format: function() {
    return this.number.toString().gsub('\\.', loom.i18n.resources['loom.format.decimalSeparator']);
  }
});

loom.format = Object.extend(loom.format, {

  /** list of configured parsers */
  dateParsers: {},

   /**
    * Parse the provided text as a date with the expected format.
    * @param text text to be parsed
    * @param format of the date: see 
    * http://www.opengroup.org/onlinepubs/007908799/xsh/strftime.html
    * @return date the parsed Date, or NaN if the text does not confirm to the expected format
    */
    parseDate: function(format, text) {
      return (loom.format.dateParsers[format] || new loom.format.DateParser(format)).parse(text);
    },
    
    /**
     * Create a date.
     * Some of the fields may be null. If the year, month or day are missing, return NaN
     */
    createDate: function(year, month, day, hour, minute, second) {
      if (year && month != null && day) { // month may be 0
            return new Date(year, month, day, hour || 0, minute || 0, second || 0);
        }
        
      // some required fields are missing
      return NaN;
    },
    
    // parse a text and return a number
    parseNumber: function(text) {
      if (text == null)
        return null;
      if (!loom.i18n.resources['loom.format.number'].test(text))
        return NaN;
      text = text.gsub('\\' + loom.i18n.resources['loom.format.groupingSeparator'], '').gsub('\\' + loom.i18n.resources['loom.format.decimalSeparator'], '.');
      return parseFloat(text);
    }
    
});

    
/**
 * A simple example of list menu behavior.
 * Note that all menu items will be expanded by default and we use javascript to un-expand them, 
 * leaving only the path to the selected item expanded.
 *
 * @author rgrocha 
 * @author icoloma 
 */
loom.ui.Menu = Class.create({
  
  /**
   * element: the container of the menu
   */
  initialize: function(element) {
  
    element.select('li a').each(function(e) {
      
      // add event listener to expand submenu on click
      var li = $(e.parentNode);
      if (li.hasClassName('submenu')) {
        e.observe('click', function(event) {
          var e = event.target;
          e.parentNode.toggleClassName('expanded');
			    if (e.getAttribute('href').startsWith('#')) {
			      event.stop();
			    }
        });
      }
      // remove all expanded attributes 
	    li.removeClassName('expanded');
    });
    
    // expand nodes in the path of the sleected menu item
    var li = element.select('li.selected').first();
    if (li) {
      li.ancestors().detect(function(e) { 
        e.addClassName('expanded');
        if (e == element)
          return true;
      });
    }
  }
  
});

/**
 * Menu to select user locale
 */
loom.ui.LocaleMenu = Class.create({

  /**
   * element: the container of the menu
   */
  initialize: function(element) {
    element.observe('click', this.onclick.bindAsEventListener(this));
  },
  
  onclick: function(event) {
    var a = event.findElement('a[hreflang]');
    if (a) {
      event.stop();
      var locale = a.getAttribute('hreflang');
      loom.cookies.set('__locale', locale, 10 * 365); // expires in ten years
      this.reload();
    }
  },
  
  /** override for tests */
  reload: function() {
    window.location.reload(true);
  }

});/**
 * Multiple file upload.
 * For details about how multiple file uploading is implemented, check 
 * UploadedFilesInterceptor.java
 * 
 * @author icoloma
 */
loom.ui.MultiUpload = Class.create({

    /**
     * @param {Element('div')} div the div where the ul and the "add new" button will be located
     */
    initialize: function(div, options) {
		this.div = div;
		
		// the name of the generated file fields
		this.inputName = this.div.getExtendedAttribute('input-name'); 
		
		options = Object.extend({
			// the template of generated file fields
			template: '<li><input type="file" name="#{name}"/><a href="#" class="removefile">#{removeFile}</a></li>\n',
			// the template of the "add new" button
			buttonTemplate: '<a href="#" class="addfile">#{addFile}</a>'
		}, options || {});
		
		Object.extend(this, options); 
		
		// add link and ul if needed
		this.ul = this.div.down('ul') || (this.div.insert('<ul/>')).down('ul');
		this.div.insert(options.buttonTemplate.interpolate({ addFile: loom.i18n.resources['loom.ui.multiupload.add'] }));
		
		// listen events: file added and file removed
		this.div.down('a.addfile').observe('click', this.addFile.bindAsEventListener(this)); 
		this.ul.observe('click', this.removeFile.bindAsEventListener(this));
      
    },

    /** adds a new input file to the form when the "add new" button is clicked */
    addFile: function(event) {
        event.stop();
        this.ul.insert(this.template.interpolate({
           name: this.getNextInputFileName(),
           removeFile: loom.i18n.resources['loom.ui.multiupload.remove']
        }));
    },
	
    /** Removes a file */
    removeFile: function(event) {
		var a = event.findElement('a.removefile');
		if (a) { 
			event.stop();
			a.up("li").remove();
		}
    },
    
    /** return the next input file name */
    getNextInputFileName: function() {
    	for (var i = 0; i < 100; i++) {
    	  var candidate = this.inputName + '[' + i + ']';
    	  var search1 = 'input[name="#{candidate}"]'.interpolate( { candidate: candidate });
    	  var search2 = 'input[name="loom-uploaded-#{candidate}"]'.interpolate( { candidate: candidate });
   	     
   	     if ($$([search1, search2]).length == 0) {
   	       return candidate;
   	     }
   	  }
   	  throw new Error('Could not find next field name (max iterations reached).');
    }
    
});

 
/**
 * Link tables where the first link of each row is used when clicking anywhere in the row. 
 *
 * @author rgrocha 
 * @author icoloma 
 */
loom.ui.tables = {

  /**
   * A table where each row is clickable and behaves as a single link
   */
  LinkTable: Class.create({
  
    initialize: function(table) {
      // add CSS class to link rows
      table.select('a').each(function(a) {
        var tr = a.up('tr');
        tr && tr.addClassName('link');
      });
      
      // listen to click event, using event delegation
      table.observe('click', this.onclick.bindAsEventListener(this));
    },
    
    onclick: function(event) {
      var tagName = event.target.tagName;
      if (tagName != 'INPUT' && tagName != 'SELECT' && tagName != 'A') { 
        var tr = event.findElement('tr');
        var a = tr == null? null : tr.down('a');
        if (a != null) {
          event.stop();
          this.redirect(a.href);
        }
      }
    },
    
    /** to override in testcases */
    redirect: function(href) {
      window.location.href = href;
    }
  
  }),
  
  /**
	 * Multiple checkbox handling routines.
	 * This class binds a set of checkboxes with a global "select/unselect all" 
	 * checkbox.
	 *
	 * To use, new MultiCheckbox(table);
	 * 
	 * This class will put an extra checkBox (a "gswitch") on the header cell of the first column
	 * where checkboxes are detected. This gswitch will check and uncheck all checkboxes at once. 
	 *
	 * @author icoloma
	 */
	MultiCheckbox: Class.create({
	
	   initialize: function(table, options) {
	     table = $(table);
	     this.checkboxes = table.select('input[type=checkbox]');
	     options = Object.extend({
	        // the position of the global switch column, starting at 0. If not set, 
	        // the first column with a checkbox will be used. In that case, if none is found
	        // (ex. the table is empty) nothing will be initialized.
	       switchColumn: -1
	     }, options || {});
	     this.options = options;
	     
	     this.column = options.switchColumn;
	     if (this.column == -1 && this.checkboxes.size() > 0) {
	       // locate the corresponding th
	       this.column = this.getColumnIndex(this.checkboxes.first());
	     }
	     
	     if (this.column != -1) {
	       // add the gswitch component
	       var th = table.select('th')[this.column];
	       th.update('<input type="checkbox" class="checkbox"/>');
	       this.gswitch = th.down('input');
	       
	       // remove checkboxes on other columns  from the list of managed checkboxes
	       this.checkboxes = this.checkboxes.findAll(function(i) { return this.getColumnIndex(i) == this.column; }.bind(this));
	       table.observe('click', this.onClick.bindAsEventListener(this));
	     }
	   },
	   
	   // return the column index of a checkbox
	   getColumnIndex: function(input) {
	     var cells = input.up('tr').select('td'); 
       return cells.indexOf(input.up('td'));
	   },
	   
	   // fired when a managed checkbox is clicked
	   onClick: function(event) {
	     var input = event.findElement('input[type=checkbox]');
	     if (input) {
	       if (input == this.gswitch) {
		         this.onSwitchClick();
	       } else if (this.checkboxes.indexOf(input) != -1) { // if it's a managed checkbox
			      this.gswitch.checked = !input.checked? false : this.checkboxes.pluck('checked');
		     }
	     }
	   },
	   
	   // fired when the global switch has been clicked
	   onSwitchClick: function() {
	     this.checkboxes.each(function(element) { element.checked = this.gswitch.checked }.bind(this) );
	   }
	   
	})

}
/**
 * Original version by Andrew Tetlaw, available at
 * http://tetlaw.id.au/view/blog/fabtabulous-simple-tabs-using-prototype/
 *
 * @author Andrew Tetlaw
 * @author icoloma 
 */
loom.ui.Tabs = Class.create({

  initialize : function(element, options) {
    this.options = Object.extend({
      selectedClass: 'selected' // css class name to apply to the selected li item
    }, options || {});
    this.container = $(element);
    this.links = this.container.select('a');
    var initTab = this.getInitialTab();
    this.show(initTab);
    this.links.without(initTab).each(this.hide.bind(this));
    this.container.observe('click', this.onClick.bind(this));
  },

  /**
   * Invoked when a tab gets clicked
   */
  onClick: function(e) {
    var a = e.findElement('a');
    if (a && this.links.indexOf(a) != -1) {
      e.stop();
	    this.show(a);
	    this.links.without(a).each(this.hide.bind(this));
    }
  },  
  
  hide : function(a) {
    var tab = this.getTabForLink(a);
    if (tab.visible) { 
	    this.container.fire('tabs:deactivate', { tab: tab.id });
      tab.hide();
      a.up('li').removeClassName(this.options.selectedClass);
    }
  },
  
  show : function(a) {
    var tab = this.getTabForLink(a);
    a.up('li').addClassName(this.options.selectedClass);
    tab.show();
    this.container.fire('tabs:activate', { tab: tab.id });
  },
  
  /**
   * Return the tab corresponding to a link
   */
  getTabForLink: function(a) {
    return $(this.getTabId(a.getAttribute('href')));
  },
  
  /**
   * return the tab ID extracted from a url anchor, null if none.
   The corresponding tab content div should have an ID with the same value
   @ href: the URL string
   */
  getTabId: function(href) {
    return href.match(/#(\w.+)/)? RegExp.$1 : null;
  },
  
  getInitialTab : function() {
    var loc = this.getTabId(document.location.href);
    var tab = null;
    if(loc) {
      tab = this.links.find(function(e) { 
        return this.getTabId(e.href) == loc; 
      }.bind(this));
    }
    return tab || this.links.first();
  }
  
});
/**
 * Form validation 
 * @author icoloma
 */
'loom.validation'.namespace();
 
loom.validation.Validator = Class.create({

    // true to allow empty input
    nullAllowed: true,

    // returns true if the element has an error message
    hasMessage: function() {
      return this.element.hasClassName("error");
    },
    
    // throw an exception with the specified error message
    error: function(message) {
     throw new loom.validation.ValidationException(message, this);
    },
    
    validate: function() {
      try {
        if (!this.element.value.blank() || !this.nullAllowed)
          this.validateImpl();
      } catch (e) {
        // consider possible message override
        var message = this.element.getExtendedAttribute(this.name + "-message");
        if (e instanceof loom.validation.ValidationException && message)
          this.error(message)
        throw e;
      }
    }

});


Object.extend(loom.validation, {

  ValidationException: Class.create({
		initialize: function(message, validator) {
			this.message = message;
			this.validator = validator;
			this.name = "ValidationException";
		}
  }),
  
  RequiredValidator: Class.create(loom.validation.Validator,  {

	  name: "required",
    
    nullAllowed: false,
	    
	  // validate a required field
	  validateImpl: function() {
	    if (this.element.value.blank()) {
	      this.error("loom.validation.requiredFailed");
	    }
	  }
    
  }),

  // validates a string 
  StringValidator: Class.create(loom.validation.Validator, {

	  initValidator: function() {
	    this.name = "string";
	    this.minLength = this.element.getExtendedAttribute('minLength');
	    var m = this.element.getAttribute('pattern');
	    if (m) 
	      this.maskPattern = new RegExp(m);
	  },
	  
	  validateImpl: function() {
	    var c = this.element;
	    var value = c.value.strip();
	    if (this.minLength != null && value.length < this.minLength)
	      this.error("loom.validation.minLengthFailed");
	    else if (this.maskPattern != null && !this.maskPattern.test(value))
	      this.error("loom.validation.maskFailed");
	  },
	  
	  // Fires when a textarea gets modified, and ensures that it does not get longer than maxlength characters
	  onTextAreaKeyPress: function(event) {
	    var evtc = event.keyCode; 
	    var c = this.element;
	  	if (c.value.length >= c.getAttribute('maxlength') && loom.ui.event.SPECIAL_KEYS.indexOf(evtc) == -1) {
	        event.stop();
	    }
	  },
	  
	  // because paste etc are not included in the "keypress" listener:
	  onTextAreaChange: function() {
	     this.element.value = this.element.value.substring(0, this.element.getAttribute('maxlength'));
	  }
  
  }),

  // validates a number
  NumberValidator: Class.create(loom.validation.Validator, {
  
    initValidator: function() {
	    var c = this.element;
		  this.name = "number";
	    this.minValue = loom.format.parseNumber(c.getAttribute('min'));
	    this.maxValue = loom.format.parseNumber(c.getAttribute('max'));
	    this.excludeMin = c.getExtendedAttributeAsBoolean('exclude-min');
	    this.excludeMax = c.getExtendedAttributeAsBoolean('exclude-max', 'true');
	    this.scale = c.getExtendedAttribute('scale');
	    this.precision = c.getExtendedAttribute('precision');
    },
    
	  validateImpl: function() {
	    var c = this.element;
	    var value = loom.format.parseNumber(c.value);
	    var numberFormat = new loom.format.NumberFormat(value);
	    if (isNaN(value))
	      this.error("loom.conversion.numberFailed");
	    else if (this.minValue != null && (this.excludeMin && value <= this.minValue || !this.excludeMin && value < this.minValue))
	      this.error("loom.validation.numberMinFailed");
	    else if (this.maxValue != null && (this.excludeMax && value >= this.maxValue || !this.excludeMax && value > this.maxValue))
	      this.error("loom.validation.numberMaxFailed");
	    else if (this.scale != null && this.scale < numberFormat.scale())
	      this.error("loom.validation.scaleFailed");
	    else if (this.precision != null && this.precision < numberFormat.precision())
	      this.error("loom.validation.precisionFailed");
	  }
  
  }),
  
  // validate a date
  DateValidator: Class.create(loom.validation.Validator, {
	  
    initValidator: function(name) {
      var c = this.element;
	    this.format = loom.i18n.resources['loom.format.jsDate'];
      this.name = "date";
	  this.minValue = c.getAttributeAsDate('min');
      this.maxValue = c.getAttributeAsDate('max');
      (this.minValue) && (this.minValue.date = this.minValue.format(this.format)); // for error messages 
      (this.maxValue) && (this.maxValue.date = this.maxValue.format(this.format));
      this.excludeMin = c.getExtendedAttributeAsBoolean('exclude-min');
      this.excludeMax = c.getExtendedAttributeAsBoolean('exclude-max', 'true');
    },
	  
	  validateImpl: function() {
		  var c = this.element;      
	    var value = loom.format.parseDate(this.format, c.value);
	    
  	    if (!value || value == NaN) 
  	      this.error("loom.conversion.dateFailed");
  	    if (this.minValue && (this.excludeMin && value <= this.minValue || !this.excludeMin && value < this.minValue))
  	      this.error("loom.validation.dateMinFailed");
  	    else if (this.maxValue && (this.excludeMax && value >= this.maxValue || !this.excludeMax && value > this.maxValue))
  	      this.error("loom.validation.dateMaxFailed");
    }
  
  }),
  
  /** 
   * get the error display element associated to this field, if any. Creates one if not found 
   */
  getErrorElement: function(element, createIfNull) {
    var label = element.up('label');
    var c = label.next();
    if (c && c.tagName == 'SPAN' && c.hasClassName('error'))
      return c;
    label.insert({ after: '<span class="error" style="display:none"></span>' });
    return label.next();
  },
  
  // clears the error message 
  clearMessage: function(element) {
    element.removeClassName("error");
    element.alt = element._alt || '';
    element.title = element._title || '';
      
    var errorelement = loom.validation.getErrorElement(element);
    errorelement? errorelement.hide().innerHTML = '' : loom.validation.removeUnboundMessage(element.identify());
  },
  
  // assigns an error message to the element
  setMessage: function(exception) {
    var message = loom.validation.translateMessage(exception);
    var element = exception.validator.element;
    
   	element.addClassName("error");
    element.alt = message;
    element.title = message;
    
    var errorelement = loom.validation.getErrorElement(element);
    if (errorelement) {
      errorelement.innerHTML = message;
      errorelement.show();
    } else {
      loom.validation.addUnboundMessage(element.identify(), message);
    }
  },
  
  
  // returns the translated message
  translateMessage: function (exception) {
    var messageTemplate = loom.i18n.resources[exception.message];
    if (messageTemplate == null)
      return exception.message;
    var element = exception.validator.element;
    // use ${} instead of #{} to reuse the same strings on the server side
    return messageTemplate.interpolate({
	      propertyName: loom.ui.getPropertyName(element), validator: exception.validator, value: element.value, length: element.value.length
	    }, /(^|.|\r|\n)(\$\{(.*?)\})/);
  },
  
  // adds an error message for a given element
  addUnboundMessage: function (elementId, message) {
    var errors = $("errors");
    if (errors == null) {
      throw new Error(message);
    }
    var id = "errors-" + elementId;
    var item = $(id);
    if (item != null) {
      item.update(message);
    } else {
      var ul = errors.firstDescendant();
	  if (!ul) {
	  	errors.insert('<ul/>');
	    ul = errors.firstDescendant();
	  }
	  ul.append('<li id="#{id}">#{message}</li>'.interpolate({id : id, message: message}));
    }
    errors.show();
  },
  
  // removes the error message of a given element
  removeUnboundMessage: function (elementId) {
    var item = $("errors-" + elementId);
    if (item != null) {
      var ul = item.up('ul');
      item.remove();
      if (ul.empty())
        $("errors").hide();
    }
  }
  
});

/**
 * Modifications of Element are kept to a minimum, to avoid collisions with other frameworks
 */
Element.addMethods({

  // adds a validator to this Element instance
  addValidator: function(element, validator) {
    validator.element = element;
    if (validator.initValidator)
      validator.initValidator();
    if (!element.validators)
      element.validators = [];
    element.validators.push(validator);
  },

  // launches all validators registered for an element
  validate: function(element) {
    try { 
      if (!element.visible() || !element.validators)
        return;
      element.validators.invoke('validate');
      loom.validation.clearMessage(element);
    } catch (e) {
      if (e instanceof loom.validation.ValidationException)
        loom.validation.setMessage(e);
      else
        throw e;
    }
  },

  // bind validations to a single element
  bindValidations: function(element) { 
	  
    if (element.hasClassName("required"))
      element.addValidator(new loom.validation.RequiredValidator());
    if (element.hasClassName("number"))
      element.addValidator(new loom.validation.NumberValidator());
    if (element.hasClassName("date")) 
      element.addValidator(new loom.validation.DateValidator());
    if (element.hasClassName("string")) {
      var validator = new loom.validation.StringValidator();
      element.addValidator(validator);
		  if (element.tagName == 'TEXTAREA' && element.getAttribute('maxlength')) { 
				element.observe('keypress', validator.onTextAreaKeyPress.bindAsEventListener(validator));
				element.observe('change', validator.onTextAreaChange.bindAsEventListener(validator));
		  }
    }
      
    element.observe("change", element.validate.bindAsEventListener(element));
    element._alt = element.alt;
    element._title  = element.title;
  }
  
});
  
Element.addMethods('FORM', {

  bindValidations: function(form) {

	  form.select('input[type=text]', 'input[type=password]', 'textarea', 'select').invoke('bindValidations');
	  
	  // select the first error element
	  form.select('input.error', 'select.error', 'textarea.error').any(Element.activate);
  }
  
});
    

