/*
 * Copyright 2002-2006 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.loom.mock;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;

import javax.servlet.ServletException;

import org.apache.commons.io.IOUtils;
import org.loom.action.Action;
import org.loom.log.Log;
import org.loom.persistence.file.PersistentFile;
import org.loom.resolution.ForwardResolution;
import org.loom.resolution.RedirectResolution;
import org.loom.resolution.Resolution;
import org.loom.servlet.CacheControl;
import org.loom.servlet.LoomServletRequest;
import org.loom.servlet.LoomServletResponse;
import org.loom.url.UrlBuilder;

/**
 * A mock Resolution class
 * @author icoloma
 *
 */
public class MockResolution implements Resolution, RedirectResolution, ForwardResolution {

	/** forward/redirect */
	private String type;
	
	/** the url to be forwarded */
	private UrlBuilder url;
	
	/** the file to be sent */
	private PersistentFile file;
	
	/** true to write the sent file to disk in a temporary file, for manual checking */
	private boolean writeFile;
	
	private CacheControl cacheControl;
	
	private static Log log = Log.getLog();
	
	private static MockResolution lastResolution;
	
	public static MockResolution getLastResolution() {
		return lastResolution;
	}
	
	public MockResolution(String type, String url) {
		this.type = type;
		this.url = new UrlBuilder();
		this.url.setRawUrl(url);
		lastResolution = this;
	}
	
	public MockResolution(String type, UrlBuilder builder) {
		this.type = type;
		this.url = builder;
		lastResolution = this;
	}

	public MockResolution(PersistentFile file, boolean writeFile) {
		this.file = file;
		this.writeFile = writeFile;
		lastResolution = this;
	}

	public MockResolution(Class<? extends Action> actionClass, String event) {
		lastResolution = this;
	}

	public void resolve(LoomServletRequest request, LoomServletResponse response) throws IOException, ServletException {
		if (url != null) {
			log.info("Resolution ", type, ": ", url.getURL(request));
		} else if (file != null) {
			log.info("Resolution file sent: [", file.getFilename(), ", content-type=", file.getContentType(), 
					", file-size=", file.getFileSize(), "bytes]");
			if (writeFile) {
				writeFile();
			}
		} else
			throw new RuntimeException("All the resolution fields are empty. Specify a url or a file!");
			
	}
	
	public String getForwardedUrl() {
		return url.getRawUrl();
	}
	
	private void writeFile() throws IOException, FileNotFoundException {
		File f = File.createTempFile("resolution", file.getFilename());
		FileOutputStream out = null;
		try {
			out = new FileOutputStream(f);
			IOUtils.copy(file.getContents().getContentsAsStream(), out);
		} finally {
			log.info("File contents are written to ", f.getAbsoluteFile());
			IOUtils.closeQuietly(out);
		}
	}

	/**
	 * Adds a parameter to the redirection url
	 */
	public RedirectResolution addParameter(String name, Object value) {
		url.addParameter(name, value);
		return this;
	}
	
	/**
	 * Adds a parameter to the redirection url
	 */
	public RedirectResolution addPropertyParameter(String property) {
		return addParameter(property, property + " value");
	}

	public RedirectResolution setHash(String anchor) {
		url.setHash(anchor);
		return this;
	}

	public String getType() {
		return type;
	}

	public UrlBuilder getUrl() {
		return url;
	}

	public PersistentFile getFile() {
		return file;
	}

	public MockResolution setCacheControl(CacheControl cacheControl) {
		this.cacheControl = cacheControl;
		return this;
	}

	public CacheControl getCacheControl() {
		return cacheControl;
	}
	

}
