/*
 * Copyright 2002-2006 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.loom.test;

import java.util.Set;

import org.loom.action.Action;
import org.loom.annotation.processor.AbstractInterceptorAnnotationProcessor;
import org.loom.annotation.processor.AnnotationProcessor;
import org.loom.annotation.processor.metadata.AnnotationMetadataFactory;
import org.loom.binding.PropertyBinderFactory;
import org.loom.interceptor.Interceptor;
import org.loom.mapping.ActionMapping;
import org.loom.mapping.ActionMappingFactory;
import org.loom.mapping.Event;
import org.loom.mapping.RestActionMappingFactory;
import org.loom.mock.MockInterceptorFactory;
import org.loom.util.TestUtil;
import org.loom.validator.PropertyValidator;

/**
 * Parent of test classes for AnnotationProcessors
 * @author icoloma
 *
 * @param <T> the AnnotationProcessor class
 */
public abstract class AbstractAnnotationProcessorTests<T extends AnnotationProcessor> {
	
	protected T processor;
	
	protected ActionMapping mapping;
	
	private ActionMappingFactory factory;
	
	/** the action class to process annotations from */
	private Class<? extends Action> actionClass;
	
	private AnnotationMetadataFactory metadataFactory = new AnnotationMetadataFactory();
	
	private PropertyBinderFactory propertyBinderFactory = new PropertyBinderFactory();
	
	/**
	 * Prepare to process annotations  for the provided Action class
	 * @param actionClass
	 */
	public void init(Class<? extends Action> actionClass) {
		// this should be removed, it's here because of a bug in ConverterFactory
		TestUtil.initConfig();
		
		this.actionClass = actionClass;
		if (processor instanceof AbstractInterceptorAnnotationProcessor) {
			((AbstractInterceptorAnnotationProcessor)processor).setInterceptorFactory(new MockInterceptorFactory());
		}
		mapping = new ActionMapping();
		mapping.setHandle("my");
		mapping.setActionClass(actionClass);
		mapping.setPropertyBinderMap(propertyBinderFactory.createPropertyBinderMap(actionClass));
		factory = new RestActionMappingFactory();
		factory.initEvents(mapping);
	}
	
	/** 
	 * @return the first validator with the required class, propertyPath and Event, or null if not found 
	 */
	@SuppressWarnings("unchecked")
	protected <V extends PropertyValidator> V getValidator(Event event, String propertyPath, Class<V> validatorClass) {
		for (PropertyValidator v : event.getValidatorsForProperty(propertyPath)) {
			if (validatorClass.isAssignableFrom(v.getClass())) {
				return (V)v;
			}
		}
		return null;
	}
	
	/** 
	 * @return the first interceptor with the required class and Event, or null if not found 
	 */
	protected <I extends Interceptor> I getInterceptor(Event event, Class<I> interceptorClass) {
		Set<I> l = event.getInterceptors().findInterceptorsWithClass(interceptorClass);
		return l.isEmpty()? null : l.iterator().next();
	}
	
	/** invoke to process annotations with the current config */
	protected void processAnnotations() {
		processor.process(metadataFactory.create(actionClass), mapping);
	}
	

}
