/*
 * Copyright 2002-2006 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.loom.test;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.util.Comparator;
import java.util.Locale;

import org.junit.Before;
import org.junit.Test;
import org.loom.config.Config;
import org.loom.converter.Converter;
import org.loom.i18n.Messages;
import org.loom.i18n.MessagesRepository;
import org.loom.i18n.MessagesRepositoryFactory;
import org.loom.log.Log;
import org.loom.servlet.params.StringParameterValue;
import org.loom.util.TestUtil;

/**
 * Parent class of Converter tests
 * @author icoloma
 *
 * @param <T> the class of the Converter
 */
public abstract class AbstractConverterTests<T extends Converter> {

	/** the Converter that is being tested */
	protected T converter;
	
	/** the MessagesRepository instance that will be used to retrieve messages */
	protected MessagesRepositoryFactory messagesRepositoryFactory;

	/** a different Locale - hey, we are a spanish factory :) */
	protected static Locale SPANISH = new Locale("es");
	
	private static Log log = Log.getLog();
	
	@Before
	public final void initConfig() {
		TestUtil.initConfig();
		messagesRepositoryFactory = Config.getInstance().getMessagesRepositoryFactory();
	}
	
	/**
	 * Check that the expected text is converted back and forth to the expected value.
	 * This method invokes both getAsText() and getAsObject()
	 * @param valueAsString The text representation of the object
	 * @param valueAsObject The expected parsed value
	 * @param locale the conversion Locale
	 */
	protected void assertConvertedText(String valueAsString, Object valueAsObject, Locale locale) {
		assertConvertedText(valueAsString, valueAsObject, locale, null);
	}
	
	/**
	 * Check that the expected text is converted back and forth to the expected value.
	 * This method invokes both getAsText() and getAsObject()
	 * @param valueAsString The text representation of the object
	 * @param valueAsObject The expected parsed value
	 * @param locale the conversion Locale
	 * @param comparator the Comparator class used to compare parsed instances, null to use equals()
	 */
	@SuppressWarnings("unchecked")
	protected void assertConvertedText(String valueAsString, Object valueAsObject, Locale locale, Comparator comparator) {
		// convert to Text
		log.info("Converting to text ", valueAsObject, " (actual) -> ", valueAsString, " (expected), locale ", locale );
		MessagesRepository repository = messagesRepositoryFactory.getRepository(locale);
		String actualText = converter.getAsText(valueAsObject, repository);
		assertEquals(valueAsString, actualText);
		
		// convert back to object
		log.info("Converting back to object ", actualText);
		Messages messages = new Messages();
		Object convertedValue = converter.getAsObject(new StringParameterValue("kk", actualText), messages, repository);
		if (comparator == null) {
			assertEquals(valueAsObject, convertedValue);
		} else {
			assertEquals("Original and converted values are not equal", 0, comparator.compare(valueAsObject, convertedValue));
		}
		if (messages.size() > 0) {
			log.info(TestUtil.toString(messages));
			fail("There have been errors trying to convert " + valueAsString);
		}
	}
	
	/**
	 * Check that the provided String will produce a conversion error.
	 * Fails if the string gets parsed without error.
	 * @param text the text to parse
	 * @param locale the locale to sue for parsing
	 */
	protected void assertConversionFails(String text, Locale locale) {
		log.info("Checking that conversion fails for ", text);
		MessagesRepository repository = messagesRepositoryFactory.getRepository(locale);
		Messages messages = new Messages();
		Object convertedValue = converter.getAsObject(new StringParameterValue("kk", text), messages, repository);
		assertNull("Expected null, but instead: " + convertedValue, convertedValue);
		assertTrue(messages.size() > 0);
		log.info(repository.translateMessage(messages.getAllMessages().get(0)));
	}

	/**
	 * Check that the converted object has the expected class
	 */
	@SuppressWarnings("unchecked")
	protected void assertReturnValue(String string, Class expectedClass) {
		MessagesRepository repository = messagesRepositoryFactory.getRepository(SPANISH);
		Messages messages = new Messages();
		Object convertedValue = converter.getAsObject(new StringParameterValue("kk", string), messages, repository);
		assertNotNull(convertedValue);
		assertTrue(expectedClass.isAssignableFrom(convertedValue.getClass()));
	}
	
	/**
	 * Test the Converter with null or empty value
	 */
	@Test
	public void testNullOrEmptyValue() {
		MessagesRepository repository = messagesRepositoryFactory.getRepository(SPANISH);
		Messages messages = new Messages();
		assertNull(converter.getAsObject(new StringParameterValue("kk", ""), messages, repository));
		assertTrue(TestUtil.toString(messages), messages.size() == 0);
		assertNull(converter.getAsObject(new StringParameterValue("kk", null), messages, repository));
		assertTrue(TestUtil.toString(messages), messages.size() == 0);
		String s = converter.getAsText(null, repository);
		assertTrue(s == null || s.length() == 0);
	}

}
