/*
 * Copyright 2002-2006 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.loom.test;

import org.loom.action.Action;
import org.loom.config.Config;
import org.loom.i18n.Messages;
import org.loom.mapping.ActionMapping;
import org.loom.mapping.ActionMappingRepository;
import org.loom.mapping.ParsedAction;
import org.loom.util.ClassUtils;

/**
 * Parent class for test cases that need a ParsedAction initialized.
 * @author icoloma
 */
public abstract class AbstractParsedActionTests extends AbstractRequestTests {

	/** the ParsedAction object */
	private ParsedAction parsedAction;
	
	/** the Action object */
	private Action action;
	
	/**
	 * Initializes the action with the provided Action class name.
	 * If the action classs is not yet registered, it gets added. 
	 * @param actionClass the action class to instantiate
	 * @param eventName the event name to use, null for the default event
	 */
	public void initActionAndEvent(Class<? extends Action> actionClass, String eventName) {

		addActionMapping(actionClass);
		action = ClassUtils.newInstance(actionClass);
		ActionMappingRepository repository = Config.getInstance().getActionMappingRepository();
		ActionMapping mapping = repository.getActionMappingByClass(actionClass);
		parsedAction = new ParsedAction(mapping, action, null, new Messages(), request, response);
		parsedAction.setEvent(mapping.getEvent(eventName));
		request.setParsedAction(parsedAction);
		action.setContext(actionContext);
		
	}
	
	/**
	 * Initializes the action with the provided Action class name
	 * If the action classs is not yet registered, it gets added. 
	 * @param actionClass the action class to instantiate
	 */
	public void initAction(Class<? extends Action> actionClass) {
		initActionAndEvent(actionClass, null);
	}

	/**
	 * Get the initialized action instance
	 * @param <T> the concrete action class expected
	 */
	@SuppressWarnings("unchecked")
	protected <T extends Action> T getAction() {
		return (T) action;
	}

	public ParsedAction getParsedAction() {
		return parsedAction;
	}
	
}
