/*
 * Copyright 2002-2006 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.loom.test;

import static org.junit.Assert.assertEquals;

import java.io.IOException;

import junit.framework.ComparisonFailure;

import org.junit.After;
import org.junit.Before;
import org.loom.action.Action;
import org.loom.action.ActionContext;
import org.loom.config.Config;
import org.loom.log.Log;
import org.loom.mapping.ActionMapping;
import org.loom.mapping.ActionMappingRepository;
import org.loom.mock.MockJspFragment;
import org.loom.mock.MockJspWriter;
import org.loom.mock.MockPageContext;
import org.loom.mock.MockResolutionFactory;
import org.loom.servlet.LoomServletRequestImpl;
import org.loom.servlet.LoomServletResponseImpl;
import org.loom.util.TestUtil;
import org.springframework.mock.web.MockHttpServletRequest;
import org.springframework.mock.web.MockHttpServletResponse;
import org.springframework.mock.web.MockServletContext;

/**
 * Parent class of any test that needs a mock servlet request object
 * properly initialized.
 * @author icoloma
 *
 */
public abstract class AbstractRequestTests {

	/** standard http objects */
	protected MockServletContext servletContext;
	protected MockHttpServletResponse mockResponse;
	protected MockHttpServletRequest mockRequest;
	
	/** loom objects */
	protected Config config;
	protected LoomServletRequestImpl request;
	protected LoomServletResponseImpl response;
	protected ActionContext actionContext;
	
	/** jsp objects */
	protected MockPageContext pageContext;
	protected MockJspFragment body;

	protected static Log log = Log.getLog();
	
	@Before
	public final void initRequest() {
		TestUtil.initConfig();
		config = Config.getInstance();
		servletContext = (MockServletContext) config.getServletContext();
		config.setServletContext(servletContext);
		
		// spring mock classes
		mockRequest = new MockHttpServletRequest(servletContext, "GET", "/webcontext/jspPage.jsp");
		mockRequest.setContextPath("/webcontext");
		mockResponse = new MockHttpServletResponse();
		
		// loom classes
		request = LoomServletRequestImpl.getOrCreateInstance(mockRequest);
		LoomServletRequestImpl.setThreadLocal(request);
		response = LoomServletResponseImpl.getInstance(request, mockResponse);
		actionContext = new ActionContext();
		actionContext.setRequest(request);
		actionContext.setResponse(response);
		actionContext.setResolutionFactory(new MockResolutionFactory());

		// jsp classes
		pageContext = new MockPageContext(servletContext, request, response);
		body = new MockJspFragment(pageContext);

	}
	
	/**
	 * Add one Action class to the current config, if not already present
	 * @param actionClass the class to be added to the current config
	 * @return the corresponding ActionMapping
	 */
	protected ActionMapping addActionMapping(Class<? extends Action> actionClass) {
		ActionMappingRepository repository = config.getActionMappingRepository();
		return repository.addActionMapping(actionClass);
	}

	
	/**
	 * Show the tag output on the standard output
	 */ 
	@After
	public void flushOutput() throws IOException {
		if (pageContext != null) {
			pageContext.getOut().flush();
		}
	}
	
	/**
	 * Check that the test output contains a certain substring
	 */
	protected void assertContains(String text) {
		if (!getOutput().contains(text)) {
			throw new ComparisonFailure("Expected text not found", text, getOutput());
		}
	}
	
	/**
	 * Check that the test output does not contain a certain substring
	 */
	protected void assertNotContains(String text) {
		if (getOutput().contains(text)) {
			throw new ComparisonFailure("Unexpected text found in output", text, getOutput());
		}
	}
	
	/**
	 * Check that the test output is exactly equals to some text
	 */
	protected void assertOutput(String text) {
		assertEquals(text, getOutput());
	}

	/**
	 * Check that the attribute has the expected value
	 */
	protected void assertAttribute(String name, Object value) {
		assertContains(name + "=\"" + value + "\"");
	}
	
	/**
	 * Check that the expected attribute exists
	 */
	protected void assertAttributePresent(String name) {
		assertContains(name + "=\"");
	}
	
	/**
	 * Check that the expected attribute does not exist
	 */
	protected void assertAttributeMissing(String name) {
		assertNotContains(name + "=\"");
	}

	/**
	 * @return the current output of the tag
	 */
	protected String getOutput() {
		return ((MockJspWriter)pageContext.getOut()).toString();
	}
	
	protected MockHttpServletRequest getMockRequest() {
		LoomServletRequestImpl req = (request);
		return (MockHttpServletRequest)req.getRequest();
	}
	


}