/*
 * Copyright 2002-2006 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.loom.util;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Locale;

import org.loom.config.ActionScanner;
import org.loom.config.Config;
import org.loom.i18n.Message;
import org.loom.i18n.Messages;
import org.loom.i18n.MessagesRepository;
import org.loom.i18n.MessagesRepositoryFactoryImpl;
import org.loom.mock.MockDependencyInjectionAdapter;
import org.loom.mock.MockResourcesWatchdog;
import org.springframework.mock.web.MockServletContext;


public class TestUtil {
	
	/**
	 * Method implemented for debugging purposes
	 */
	public static String toString(Messages messages) {
		MessagesRepository repository = Config.getInstance().getMessagesRepositoryFactory().getRepository(Locale.getDefault());
		StringBuilder builder = new StringBuilder();
		for (Message m : messages.getAllMessages())
			builder.append(repository.translateMessage(m)).append("\n");
		return builder.toString();
	}
	
	/**
	 * Serializes and deserializes an object
	 */
	@SuppressWarnings("unchecked")
	public static <T extends Serializable> T serializeAndDeserialize(T instance) throws IOException, ClassNotFoundException {
		ByteArrayOutputStream bos = new ByteArrayOutputStream();
		ObjectOutputStream out = new ObjectOutputStream(bos);
		out.writeObject(instance);
		byte[] serialized = bos.toByteArray();
		ByteArrayInputStream bis = new ByteArrayInputStream(serialized);
		ObjectInputStream input = new ObjectInputStream(bis);
		T deserialized = (T) input.readObject();
		return deserialized;
	}


	/**
	 * Initialize the Config instance if needed
	 */
	public static void initConfig() {
		if (Config.getInstance() == null) {
			Config config = new Config();
			Config.setInstance(config);
			config.setServletContext(new MockServletContext());
			
			ActionScanner scanner = new ActionScanner();
			scanner.setActionPackages("org.loom.action");
			config.setActionScanner(scanner);
			config.setDependencyInjectionAdapter(new MockDependencyInjectionAdapter());
			config.setResourcesWatchdog(new MockResourcesWatchdog());
			
			config.validate();
			config.scanAndConfigureActions();
			
			// consistent behavior of tests, no matter the O.S. locale
			MessagesRepositoryFactoryImpl factory = (MessagesRepositoryFactoryImpl)config.getMessagesRepositoryFactory();
			factory.setDefaultLocale(Locale.ENGLISH);
			factory.setLocations(Arrays.asList("classpath:resources/loom-messages", "classpath:resources/messages"));
			
		}
	}
	
}
