/*******************************************************************************
 * Copyright (c) Faktor Zehn GmbH - faktorzehn.org
 * 
 * This source code is available under the terms of the AGPL Affero General Public License version
 * 3.
 * 
 * Please see LICENSE.txt for full license terms, including the additional permissions and
 * restrictions as well as the possibility of alternative license terms.
 *******************************************************************************/

package org.faktorips.codegen;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;

import org.faktorips.util.StringUtil;

/**
 * An ImportDeclaration is an ordered set of import statements.
 * <p>
 * When adding new import statements it is checked that no unnecessary statements are added, for
 * example if you add <code>java.util.ArrayList</code> and then <code>java.util.*</code> only the
 * latter statement is kept. Also import statements for classes residing in <code>java.lang</code>
 * and for primitive types are ignored.
 * 
 * @author Jan Ortmann
 */
public class ImportDeclaration {

    private static final String JAVA_LANG_ASTERIX = "java.lang.*"; //$NON-NLS-1$

    /** List that holds the class imports. */
    private List<String> classes;

    /** Set that holds the package imports (e.g. java.util.*). */
    private List<String> packages;

    /**
     * Creates a new import declaration.
     */
    public ImportDeclaration() {
        classes = new ArrayList<>();
        packages = new ArrayList<>();
    }

    /**
     * Copy constructor.
     */
    public ImportDeclaration(ImportDeclaration decl) {
        this();
        add(decl);
    }

    /**
     * Constructs a new import declaration that contains all import statements from the given
     * declaration that are not covered by the package.
     */
    public ImportDeclaration(ImportDeclaration decl, String packageName) {
        this();
        String packageImport = packageName + ".*"; //$NON-NLS-1$
        ImportDeclaration temp = new ImportDeclaration(decl);
        temp.add(packageImport);
        for (Iterator<String> it = temp.iterator(); it.hasNext();) {
            String importSpec = it.next();
            if (!importSpec.equals(packageImport)) {
                add(importSpec);
            }
        }
    }

    /**
     * Returns true if this is a package import, e.g. <code>java.util.*</code> Returns false if
     * importSpec is null.
     */
    public static final boolean isPackageImport(String importSpec) {
        if (importSpec == null) {
            return false;
        }
        return "*".equals(importSpec.substring(importSpec.length() - 1)); //$NON-NLS-1$
    }

    /**
     * Adds the class to the import list.
     */
    public void add(Class<?> clazz) {
        add(clazz.getName());
    }

    /**
     * Adds all imports in the given import declaration to this declaration. Does nothing if the
     * given import decl is null.
     */
    public void add(ImportDeclaration decl) {
        if (decl == null) {
            return;
        }
        for (Iterator<String> it = decl.iterator(); it.hasNext();) {
            add(it.next());
        }
    }

    /**
     * Adds the import specifications to the list of imports. Does nothing if the given importSpecs
     * is null.
     */
    public void add(String[] importSpecs) {
        if (importSpecs == null) {
            return;
        }
        for (String importSpec : importSpecs) {
            add(importSpec);
        }
    }

    /**
     * Adds the import specification to the list of imports. Does nothing if the given importSpecs
     * is null.
     */
    public void add(String importSpec) {
        if (importSpec == null || isCovered(importSpec)) {
            return;
        }
        if (isPackageImport(importSpec)) {
            removeClassImports(importSpec);
            packages.add(importSpec);
        } else {
            classes.add(importSpec);
        }
    }

    /**
     * Removes the class imports that are covered by the package import.
     */
    private void removeClassImports(String packageImport) {
        for (Iterator<String> it = classes.iterator(); it.hasNext();) {
            String classImport = it.next();
            if (classImportCoveredByPackageImport(classImport, packageImport)) {
                it.remove();
            }
        }
    }

    /**
     * Returns true if the class is covered by the import declaration. That is, if either an import
     * for that class exists or the package the class resides in is imported.
     * 
     * @throws NullPointerException if clazz is null.
     */
    public boolean isCovered(Class<?> clazz) {
        return isCovered(clazz.getName());
    }

    /**
     * Returns true if the import specification is covered by this import declaration.
     * 
     * @throws NullPointerException if importSpec is null.
     */
    public boolean isCovered(String importSpec) {
        if (importSpec == null) {
            throw new NullPointerException();
        }
        // CSOFF: BooleanExpressionComplexity
        if (importSpec.equals(Boolean.TYPE.getName()) || importSpec.equals(Integer.TYPE.getName())
                || importSpec.equals(Double.TYPE.getName()) || importSpec.equals(Long.TYPE.getName())
                || "void".equals(importSpec)) { //$NON-NLS-1$
            // CSON: BooleanExpressionComplexity
            // this is a primitive type
            return true;
        }
        // CSON: BooleanExpressionComplexity
        if (isPackageImport(importSpec)) {
            if (JAVA_LANG_ASTERIX.equals(importSpec)) {
                return true;
            }
            return packages.contains(importSpec);
        }
        if (classes.contains(importSpec)) {
            return true;
        }
        return isCovered(StringUtil.getPackageName(importSpec) + ".*"); //$NON-NLS-1$
    }

    /**
     * Returns true if the import specification is already covered by an existing import
     * specification.
     */
    private boolean classImportCoveredByPackageImport(String classImport, String packageImport) {
        return (StringUtil.getPackageName(classImport) + ".*").equals(packageImport); //$NON-NLS-1$
    }

    /**
     * Returns an Iterator over the import statements as Strings.
     */
    public Iterator<String> iterator() {
        return getImports().iterator();
    }

    public Set<String> getImports() {
        Set<String> allImports = new LinkedHashSet<>();
        allImports.addAll(packages);
        allImports.addAll(classes);
        return allImports;
    }

    /**
     * Returns the number of imports.
     */
    public int getNoOfImports() {
        return classes.size() + packages.size();
    }

    /**
     * Returns those imports in the <code>importsToTest</code> declaration that are not covered this
     * one. Returns an empty import declaration if either all imports are covered or importsToTest
     * is <code>null</code>.
     */
    public ImportDeclaration getUncoveredImports(ImportDeclaration importsToTest) {
        ImportDeclaration uncovered = new ImportDeclaration();
        if (importsToTest == null) {
            return uncovered;
        }
        for (Iterator<String> it = importsToTest.iterator(); it.hasNext();) {
            String importToTest = it.next();
            if (!isCovered(importToTest)) {
                uncovered.add(importToTest);
            }
        }
        return uncovered;
    }

    @Override
    public int hashCode() {
        return Objects.hash(classes, packages);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (!(obj instanceof ImportDeclaration)) {
            return false;
        }
        ImportDeclaration other = (ImportDeclaration)obj;
        return Objects.equals(classes, other.classes)
                && Objects.equals(packages, other.packages);
    }

    /**
     * Returns the import statements as a string. The import statements are separated by a line
     * separator. Each line has a trailing "import " and ends with a semicolon (;).
     */
    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        String separator = System.lineSeparator();
        for (Iterator<String> it = iterator(); it.hasNext();) {
            sb.append(("import ")); //$NON-NLS-1$
            sb.append(it.next());
            sb.append(";"); //$NON-NLS-1$
            sb.append(separator);
        }
        return sb.toString();
    }

}
