/*******************************************************************************
 * Copyright (c) Faktor Zehn GmbH - faktorzehn.org
 * 
 * This source code is available under the terms of the AGPL Affero General Public License version
 * 3.
 * 
 * Please see LICENSE.txt for full license terms, including the additional permissions and
 * restrictions as well as the possibility of alternative license terms.
 *******************************************************************************/

package org.faktorips.runtime.model.annotation;

import java.lang.annotation.Annotation;
import java.lang.reflect.AnnotatedElement;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;

import org.faktorips.runtime.caching.Memoizer;

/**
 * Helper class that represents a type that was generated by Faktor-IPS with meta data in its
 * annotations. The meta data for a type is split into interface and implementation, sometimes only
 * the implementation class exists. AnnotatedDeclaration allows reading the meta data annotations
 * regardless of which class was annotated in the end.
 * 
 */
public class AnnotatedDeclaration {

    private static final Memoizer<Class<?>, AnnotatedDeclaration> ANNOTATED_TYPE_CACHE = Memoizer
            .of(AnnotatedDeclaration.class, AnnotatedDeclaration::new);

    private final List<Class<?>> annotatedClasses = new ArrayList<>();

    private AnnotatedDeclaration(Class<?> modelClass) {
        add(getPublishedInterface(modelClass));
        add(getImplementationClass(modelClass));
    }

    private void add(Class<?> annotatedClass) {
        if (annotatedClass != null) {
            annotatedClasses.add(annotatedClass);
        }
    }

    /**
     * Creates a new {@link AnnotatedDeclaration} from the given class. The given class is either an
     * implementation or a published interface.
     * 
     * @param generatedClass The class that was generated for the model type
     * @return An instance of {@link AnnotatedDeclaration} for the given generated class
     */
    public static AnnotatedDeclaration from(Class<?> generatedClass) {
        try {
            return ANNOTATED_TYPE_CACHE.compute(generatedClass);
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new RuntimeException(e);
        }
    }

    /**
     * Returns <code>true</code> if this represented type is annotated with the specified annotation
     * class.
     * 
     * 
     * @param annotationClass The annotation class that should be present
     * @return <code>true</code> if the annotation is present else <code>false</code>
     */
    public boolean is(Class<? extends Annotation> annotationClass) {
        for (Class<?> annotatedClass : annotatedClasses) {
            if (annotatedClass.isAnnotationPresent(annotationClass)) {
                return true;
            }
        }
        return false;
    }

    /**
     * Returns the specified annotation if present. Returns <code>null</code> if it is not present.
     * 
     * Use #is(Class) to check whether the annotation is present or not
     * 
     * @param annotationClass The class of the requested annotation
     * @return The requested annotation if present else null
     */
    public <T extends Annotation> T get(Class<T> annotationClass) {
        for (Class<?> annotatedClass : annotatedClasses) {
            if (annotatedClass.isAnnotationPresent(annotationClass)) {
                return annotatedClass.getAnnotation(annotationClass);
            }
        }
        return null;
    }

    /**
     * Returns all fields that are declared in this type. That means it collects all fields from the
     * implementation and published interface. If this type represents a product component type it
     * also adds all fields from generations.
     * 
     * @return A list of fields that are declared in this type
     */
    public List<Field> getDeclaredFields() {
        ArrayList<Field> result = new ArrayList<>();
        for (Class<?> annotatedClass : annotatedClasses) {
            result.addAll(Arrays.asList(annotatedClass.getDeclaredFields()));
        }
        return result;
    }

    /**
     * Returns all methods that are declared in this type. That means it collects all methods from
     * the implementation and published interface. If this type represents a product component type
     * it also adds all methods from generations.
     * 
     * @return A list of fields that are declared in this type
     */
    public List<Method> getDeclaredMethods() {
        ArrayList<Method> result = new ArrayList<>();
        for (Class<?> annotatedClass : annotatedClasses) {
            for (Method declaredMethod : annotatedClass.getDeclaredMethods()) {
                // filter out overridden methods generated when overriding methods use covariant
                // return types
                if (!declaredMethod.isBridge()) {
                    result.add(declaredMethod);
                }
            }
        }
        return result;
    }

    public List<AnnotatedElement> getDeclaredElements() {
        ArrayList<AnnotatedElement> result = new ArrayList<>();
        result.addAll(getDeclaredFields());
        result.addAll(getDeclaredMethods());
        return result;
    }

    /**
     * Returns the {@link ClassLoader} from the this annotated type. This {@link ClassLoader} could
     * be used to load further resources like corresponding property files.
     * 
     */
    public ClassLoader getClassLoader() {
        return annotatedClasses.get(0).getClassLoader();
    }

    /**
     * Returns the implementation class of this type.
     */
    public Class<?> getImplementationClass() {
        return getImplementationClass(annotatedClasses.get(0));
    }

    /**
     * Returns the published interface of this type.
     */
    public Class<?> getPublishedInterface() {
        return getPublishedInterface(annotatedClasses.get(0));
    }

    public String getDeclarationClassName() {
        return getPublishedInterface() != null ? getPublishedInterface().getCanonicalName()
                : getImplementationClass()
                        .getCanonicalName();
    }

    @Override
    public int hashCode() {
        return Objects.hash(annotatedClasses);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if ((obj == null) || (getClass() != obj.getClass())) {
            return false;
        }
        AnnotatedDeclaration other = (AnnotatedDeclaration)obj;
        return Objects.equals(annotatedClasses, other.annotatedClasses);
    }

    static Class<?> getPublishedInterface(Class<?> modelObjectClass) {
        if (modelObjectClass.isInterface()) {
            return modelObjectClass;
        } else {
            Class<?>[] interfaces = modelObjectClass.getInterfaces();
            for (Class<?> iface : interfaces) {
                if (iface.isAnnotationPresent(IpsPublishedInterface.class)) {
                    return iface;
                }
            }
            return null;
        }
    }

    static Class<?> getImplementationClass(Class<?> modelObjectClass) {
        if (modelObjectClass.isInterface()) {
            if (modelObjectClass.isAnnotationPresent(IpsPublishedInterface.class)) {
                return modelObjectClass.getAnnotation(IpsPublishedInterface.class).implementation();
            } else {
                return null;
            }
        } else {
            return modelObjectClass;
        }
    }

}
