/*******************************************************************************
 * Copyright (c) Faktor Zehn GmbH - faktorzehn.org
 * 
 * This source code is available under the terms of the AGPL Affero General Public License version
 * 3.
 * 
 * Please see LICENSE.txt for full license terms, including the additional permissions and
 * restrictions as well as the possibility of alternative license terms.
 *******************************************************************************/
package org.faktorips.runtime.productdata.jpa.provider;

import java.io.IOException;
import java.io.InputStream;
import java.util.Objects;

import javax.persistence.EntityManager;

import org.faktorips.runtime.internal.toc.CustomTocEntryObject;
import org.faktorips.runtime.internal.toc.EnumContentTocEntry;
import org.faktorips.runtime.internal.toc.GenerationTocEntry;
import org.faktorips.runtime.internal.toc.IReadonlyTableOfContents;
import org.faktorips.runtime.internal.toc.ProductCmptTocEntry;
import org.faktorips.runtime.internal.toc.TableContentTocEntry;
import org.faktorips.runtime.internal.toc.TestCaseTocEntry;
import org.faktorips.runtime.internal.toc.TocEntry;
import org.faktorips.runtime.internal.toc.TocEntryObject;
import org.faktorips.runtime.productdata.jpa.commons.TocVersionEntity;
import org.faktorips.runtime.productdata.jpa.commons.TocVersionEntity.Status;
import org.faktorips.runtime.productdata.jpa.toc.DbTableOfContents;
import org.faktorips.runtime.productdata.jpa.toc.DbTocEntryAdapter;
import org.faktorips.runtime.productdataprovider.AbstractProductDataProvider;
import org.faktorips.runtime.productdataprovider.DataModifiedException;
import org.faktorips.runtime.productdataprovider.IProductDataProvider;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.xml.sax.SAXException;

/**
 * {@link IProductDataProvider} using JPA to read product data from a database.
 */
public class DbProductDataProvider extends AbstractProductDataProvider {

    private final ProductDatabase productDatabase;

    private final DbTableOfContents toc;
    private final String version;
    private final TocVersionEntity tocVersion;

    /**
     * Creates a {@link DbProductDataProvider} for a table of content identified by {@code name} and
     * {@code version}. The given {@link EntityManager} is used to read the
     * {@link DbTableOfContents} from the database and for up-to-date checks in
     * {@link #getBaseVersion()}.
     * 
     * @param name the name identifying a table of contents
     * @param modelVersion the version of the model matching the product data contained in the table
     *            of contents
     * @param productDatabase the {@link ProductDatabaseBean} used for up-to-date checks in
     *            {@link #getBaseVersion()}
     */
    public DbProductDataProvider(String name, String modelVersion, ProductDatabase productDatabase) {
        this(productDatabase.getActiveTocVersion(name, modelVersion), productDatabase);
    }

    /**
     * Creates a {@link DbProductDataProvider} for a given {@link TocVersionEntity}.
     * 
     * @param tocVersion the {@link TocVersionEntity} read from the database
     * @param productDatabase the {@link ProductDatabaseBean} used for up-to-date checks in
     *            {@link #getBaseVersion()}
     */
    public DbProductDataProvider(TocVersionEntity tocVersion, ProductDatabase productDatabase) {
        super(Objects::equals);
        this.tocVersion = tocVersion;
        this.productDatabase = productDatabase;

        toc = new DbTableOfContents(tocVersion);
        version = tocVersion.getVersion();
    }

    @Override
    public InputStream getEnumContentAsStream(EnumContentTocEntry enumContentTocEntry)
            throws DataModifiedException {
        return getResourceAsStream(enumContentTocEntry);
    }

    private InputStream getResourceAsStream(TocEntry tocEntry) throws DataModifiedException {
        if (tocEntry instanceof DbTocEntryAdapter) {
            InputStream xmlContent = ((DbTocEntryAdapter)tocEntry).getXmlContent();
            throwExceptionIfModified(tocEntry, getBaseVersion());
            return xmlContent;
        } else {
            throw new IllegalArgumentException(tocEntry + " is no " + DbTocEntryAdapter.class.getName());
        }
    }

    private void throwExceptionIfModified(TocEntry tocEntry, String currentVersion)
            throws DataModifiedException {
        if (!getVersionChecker().isCompatibleVersion(getVersion(), currentVersion)) {
            throw new DataModifiedException(MODIFIED_EXCEPTION_MESSAGE + getNameForExceptionText(tocEntry),
                    getVersion(),
                    currentVersion);
        }
    }

    private String getNameForExceptionText(TocEntry tocEntry) {
        if (tocEntry instanceof TocEntryObject) {
            return ((TocEntryObject)tocEntry).getIpsObjectId();
        } else if (tocEntry instanceof GenerationTocEntry) {
            return ((GenerationTocEntry)tocEntry).getParent().getIpsObjectId();
        } else {
            return tocEntry.toString();
        }
    }

    @Override
    public Element getProductCmptData(ProductCmptTocEntry tocEntry) throws DataModifiedException {
        return getDocumentElement(tocEntry);
    }

    private Element getDocumentElement(TocEntry tocEntry) throws DataModifiedException {
        try (InputStream inputStream = getResourceAsStream(tocEntry)) {
            Document doc = getDocumentBuilder().parse(inputStream);
            Element element = doc.getDocumentElement();
            if (element == null) {
                throw new RuntimeException("Xml resource for '" + tocEntry + "' hasn't got a document element.");
            }
            return element;
        } catch (SAXException | IOException e) {
            throw new RuntimeException("Cannot parse xml resource for '" + tocEntry + "'", e);
        }
    }

    @Override
    public Element getProductCmptGenerationData(GenerationTocEntry tocEntry)
            throws DataModifiedException {
        return getDocumentElement(tocEntry);
    }

    @Override
    public InputStream getTableContentAsStream(TableContentTocEntry tocEntry)
            throws DataModifiedException {
        return getResourceAsStream(tocEntry);

    }

    @Override
    public Element getTestcaseElement(TestCaseTocEntry arg0) throws DataModifiedException {
        throw new UnsupportedOperationException(CustomTocEntryObject.class.getName() + " is not supported in a "
                + DbProductDataProvider.class + " at the moment.");
    }

    @Override
    public IReadonlyTableOfContents getToc() {
        return toc;
    }

    @Override
    public <T> Element getTocEntryData(CustomTocEntryObject<T> arg0) throws DataModifiedException {
        throw new UnsupportedOperationException(CustomTocEntryObject.class.getName() + " is not supported in a "
                + DbProductDataProvider.class + " at the moment.");
    }

    /**
     * {@inheritDoc}
     * <p>
     * The initial {@link TocVersionEntity#getVersion() TocVersion's Version}.
     */
    @Override
    public String getVersion() {
        return version;
    }

    /**
     * {@inheritDoc}
     * <p>
     * The {@link TocVersionEntity#getVersion() version} of the {@link TocVersionEntity} for the
     * same {@link TocVersionEntity#getName() name} and {@link TocVersionEntity#getModelVersion()
     * modelVersion} as the initial {@link TocVersionEntity}, currently marked as
     * {@link Status#ACTIVE}.
     */
    @Override
    public String getBaseVersion() {
        return productDatabase.getActiveVersionNumber(tocVersion.getName(), tocVersion.getModelVersion());
    }
}
