/*******************************************************************************
 * Copyright (c) Faktor Zehn GmbH - faktorzehn.org
 * 
 * This source code is available under the terms of the AGPL Affero General Public License version
 * 3.
 * 
 * Please see LICENSE.txt for full license terms, including the additional permissions and
 * restrictions as well as the possibility of alternative license terms.
 *******************************************************************************/
package org.faktorips.runtime.productdata.jpa.toc;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.faktorips.runtime.internal.toc.CustomTocEntryObject;
import org.faktorips.runtime.internal.toc.EnumContentTocEntry;
import org.faktorips.runtime.internal.toc.EnumXmlAdapterTocEntry;
import org.faktorips.runtime.internal.toc.IReadonlyTableOfContents;
import org.faktorips.runtime.internal.toc.ModelTypeTocEntry;
import org.faktorips.runtime.internal.toc.ProductCmptTocEntry;
import org.faktorips.runtime.internal.toc.TableContentTocEntry;
import org.faktorips.runtime.internal.toc.TestCaseTocEntry;
import org.faktorips.runtime.model.IpsModel;
import org.faktorips.runtime.productdata.jpa.commons.EnumContentTocEntryEntity;
import org.faktorips.runtime.productdata.jpa.commons.ProductCmptTocEntryEntity;
import org.faktorips.runtime.productdata.jpa.commons.TableContentTocEntryEntity;
import org.faktorips.runtime.productdata.jpa.commons.TocEntryEntity;
import org.faktorips.runtime.productdata.jpa.commons.TocVersionEntity;

import edu.umd.cs.findbugs.annotations.CheckForNull;

public class DbTableOfContents implements IReadonlyTableOfContents {

    private String productDataVersion;
    private Map<String, DbEnumContentTocEntry> enumContentEntriesByEnumClass;
    private Map<String, DbTableContentTocEntry> tableContentEntriesByTableClass;
    private Map<String, DbTableContentTocEntry> tableContentEntriesByName;
    private Map<String, DbProductCmptTocEntry> productCmptEntriesByIpsObjectId;
    private Map<String, List<DbProductCmptTocEntry>> productCmptEntriesByKindIdAndVersionId;

    public DbTableOfContents(TocVersionEntity tocVersion) {
        productDataVersion = tocVersion.getModelVersion() + '-' + tocVersion.getVersion();
        Map<Class<? extends TocEntryEntity>, List<TocEntryEntity>> entriesByClass = tocVersion.getEntries().stream()
                .collect(Collectors.groupingBy(TocEntryEntity::getClass));
        enumContentEntriesByEnumClass = entriesByClass(entriesByClass, EnumContentTocEntryEntity.class)
                .map(DbEnumContentTocEntry::new)
                .collect(Collectors.toConcurrentMap(DbEnumContentTocEntry::getImplementationClassName,
                        Function.identity()));
        tableContentEntriesByName = entriesByClass(entriesByClass, TableContentTocEntryEntity.class)
                .map(DbTableContentTocEntry::new).collect(
                        Collectors.toConcurrentMap(DbTableContentTocEntry::getIpsObjectQualifiedName,
                                Function.identity()));
        tableContentEntriesByTableClass = tableContentEntriesByName.values().stream().collect(Collectors
                .toConcurrentMap(DbTableContentTocEntry::getImplementationClassName, Function.identity(),
                        // see
                        // org.faktorips.runtime.internal.toc.ReadonlyTableOfContents.internalAddEntry(TocEntryObject)
                        // for table structures with multiple contents this map is pointless, so it
                        // doesn't
                        // matter which content is saved.
                        (a, b) -> a));
        productCmptEntriesByIpsObjectId = entriesByClass(entriesByClass, ProductCmptTocEntryEntity.class)
                .map(DbProductCmptTocEntry::new)
                .collect(Collectors.toConcurrentMap(DbProductCmptTocEntry::getIpsObjectId,
                        Function.identity()));
        productCmptEntriesByKindIdAndVersionId = productCmptEntriesByIpsObjectId.values().stream()
                .collect(Collectors.groupingBy(DbProductCmptTocEntry::getKindId, ConcurrentHashMap::new,
                        Collectors.toList()));
    }

    private static <T extends TocEntryEntity> Stream<T> entriesByClass(
            Map<Class<? extends TocEntryEntity>, List<TocEntryEntity>> entriesByClass,
            Class<T> tocEntryClass) {
        return entriesByClass.getOrDefault(tocEntryClass, List.of()).stream()
                .map(tocEntryClass::cast);
    }

    @CheckForNull
    @Override
    public ProductCmptTocEntry getProductCmptTocEntry(@CheckForNull String id) {
        return id == null ? null : productCmptEntriesByIpsObjectId.get(id);
    }

    @CheckForNull
    @Override
    public ProductCmptTocEntry getProductCmptTocEntry(@CheckForNull String kindId, @CheckForNull String versionId) {
        return getVersionEntries(kindId).flatMap(
                vs -> vs.stream()
                        .filter(v -> Objects.equals(versionId, v.getVersionId()))
                        .findFirst())
                .orElse(null);
    }

    private Optional<List<DbProductCmptTocEntry>> getVersionEntries(@CheckForNull String kindId) {
        return kindId == null ? Optional.empty()
                : Optional.ofNullable(productCmptEntriesByKindIdAndVersionId.get(kindId));
    }

    @Override
    public List<ProductCmptTocEntry> getProductCmptTocEntries() {
        return new ArrayList<>(productCmptEntriesByIpsObjectId.values());
    }

    @Override
    public List<ProductCmptTocEntry> getProductCmptTocEntries(@CheckForNull String kindId) {
        return new ArrayList<>(getVersionEntries(kindId).orElseGet(Collections::emptyList));
    }

    @Override
    public List<TableContentTocEntry> getTableTocEntries() {
        return new ArrayList<>(tableContentEntriesByTableClass.values());
    }

    // javadoc does not recognize <s>/<strike>/<del> as valid html elements
    /**
     * <div style="text-decoration:line-through"> {@inheritDoc} </div>
     * 
     * @deprecated Not (yet) supported in {@link DbTableOfContents}. Please submit a feature request
     *                 if you need this.
     */
    @Deprecated
    @Override
    public List<TestCaseTocEntry> getTestCaseTocEntries() {
        return List.of();
    }

    /**
     * <div style="text-decoration:line-through"> {@inheritDoc} </div>
     * 
     * @deprecated Not (yet) supported in {@link DbTableOfContents}. Please submit a feature request
     *                 if you need this.
     */
    @Deprecated
    @CheckForNull
    @Override
    public TestCaseTocEntry getTestCaseTocEntryByQName(@CheckForNull String qName) {
        return null;
    }

    /**
     * {@inheritDoc}
     * <p>
     * <em>If the table structure allows multiple table contents any one of those contents could be
     * returned.</em>
     */
    @CheckForNull
    @Override
    public TableContentTocEntry getTableTocEntryByClassname(@CheckForNull String implementationClass) {
        return implementationClass == null ? null : tableContentEntriesByTableClass.get(implementationClass);
    }

    @CheckForNull
    @Override
    public TableContentTocEntry getTableTocEntryByQualifiedTableName(@CheckForNull String qualifiedTableName) {
        return qualifiedTableName == null ? null : tableContentEntriesByName.get(qualifiedTableName);
    }

    /**
     * <div style="text-decoration:line-through"> {@inheritDoc} </div>
     * 
     * @deprecated Not supported in {@link DbTableOfContents}. Use {@link IpsModel} instead.
     */
    @Deprecated
    @Override
    public Set<ModelTypeTocEntry> getModelTypeTocEntries() {
        return Set.of();
    }

    @Override
    public List<EnumContentTocEntry> getEnumContentTocEntries() {
        return new ArrayList<>(enumContentEntriesByEnumClass.values());
    }

    @CheckForNull
    @Override
    public EnumContentTocEntry getEnumContentTocEntry(@CheckForNull String className) {
        return className == null ? null : enumContentEntriesByEnumClass.get(className);
    }

    /**
     * <div style="text-decoration:line-through"> {@inheritDoc} </div>
     * 
     * @deprecated Not supported in {@link DbTableOfContents} because the adapters have to be
     *                 registered in model projects, not product projects.
     */
    @Deprecated
    @Override
    public Set<EnumXmlAdapterTocEntry> getEnumXmlAdapterTocEntries() {
        return Set.of();
    }

    @Override
    public String getProductDataVersion() {
        return productDataVersion;
    }

    /**
     * <div style="text-decoration:line-through"> {@inheritDoc} </div>
     * 
     * @deprecated Not (yet) supported in {@link DbTableOfContents}. Please submit a feature request
     *                 if you need this.
     */
    @Deprecated
    @CheckForNull
    @Override
    public <T> CustomTocEntryObject<T> getCustomTocEntry(@CheckForNull Class<T> type,
            @CheckForNull String ipsObjectQualifiedName) {
        return null;
    }

}
