/*******************************************************************************
 * Copyright (c) Faktor Zehn GmbH - faktorzehn.org
 * 
 * This source code is available under the terms of the AGPL Affero General Public License version
 * 3.
 * 
 * Please see LICENSE.txt for full license terms, including the additional permissions and
 * restrictions as well as the possibility of alternative license terms.
 *******************************************************************************/
package org.faktorips.runtime.productdata.jpa.toc;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;

import org.faktorips.runtime.internal.toc.TocEntry;
import org.faktorips.runtime.productdata.jpa.commons.ContentContainer;
import org.faktorips.runtime.productdata.jpa.commons.ContentEntity;
import org.faktorips.runtime.productdata.jpa.commons.TocEntryEntity;

/**
 * Wrapper for {@link TocEntry runtime ToC Entries} whose XML content should be provided by a
 * {@link ContentContainer} that is loaded from a database.
 */
public interface DbTocEntryAdapter {

    /**
     * The {@link ContentContainer} that should be used to extract the XML content.
     * 
     * @return the {@link ContentContainer} entity read from database
     */
    ContentContainer getContentContainer();

    /**
     * Extract the XML content of a ToC Entry from the entity. By default, the
     * {@link TocEntryEntity#getXmlResource() XML resource} of the entity is used which is expected
     * to be lazy loaded.
     * 
     * @return {@link ContentEntity#getXml() XML} of the {@link TocEntryEntity#getXmlResource()
     *             TocEntry's XML resource} as an {@link InputStream}
     */
    default InputStream getXmlContent() {
        return new ByteArrayInputStream(
                getContentContainer().getXmlResource().getXml().getBytes(StandardCharsets.UTF_8));
    }

}