/*
 * Decompiled with CFR 0.152.
 */
package org.fcrepo.kernel.impl;

import java.time.Duration;
import java.time.Instant;
import java.util.concurrent.Phaser;
import org.fcrepo.common.db.DbTransactionExecutor;
import org.fcrepo.common.lang.CheckedRunnable;
import org.fcrepo.kernel.api.ContainmentIndex;
import org.fcrepo.kernel.api.Transaction;
import org.fcrepo.kernel.api.TransactionState;
import org.fcrepo.kernel.api.cache.UserTypesCache;
import org.fcrepo.kernel.api.exception.RepositoryRuntimeException;
import org.fcrepo.kernel.api.exception.TransactionClosedException;
import org.fcrepo.kernel.api.exception.TransactionRuntimeException;
import org.fcrepo.kernel.api.identifiers.FedoraId;
import org.fcrepo.kernel.api.lock.ResourceLockManager;
import org.fcrepo.kernel.api.observer.EventAccumulator;
import org.fcrepo.kernel.api.services.MembershipService;
import org.fcrepo.kernel.api.services.ReferenceService;
import org.fcrepo.kernel.impl.TransactionManagerImpl;
import org.fcrepo.persistence.api.PersistentStorageSession;
import org.fcrepo.search.api.SearchIndex;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TransactionImpl
implements Transaction {
    private static final Logger log = LoggerFactory.getLogger(TransactionImpl.class);
    private final String id;
    private final TransactionManagerImpl txManager;
    private TransactionState state;
    private boolean shortLived = true;
    private Instant expiration;
    private boolean expired = false;
    private String baseUri;
    private String userAgent;
    private final Duration sessionTimeout;
    private final Phaser operationPhaser;

    protected TransactionImpl(String id, TransactionManagerImpl txManager, Duration sessionTimeout) {
        if (id == null || id.isEmpty()) {
            throw new IllegalArgumentException("Transaction id should not be empty!");
        }
        this.id = id;
        this.txManager = txManager;
        this.sessionTimeout = sessionTimeout;
        this.expiration = Instant.now().plus(sessionTimeout);
        this.state = TransactionState.OPEN;
        this.operationPhaser = new Phaser();
    }

    public synchronized void commit() {
        if (this.state == TransactionState.COMMITTED) {
            return;
        }
        this.failIfNotOpen();
        this.failIfExpired();
        this.updateState(TransactionState.COMMITTING);
        log.debug("Waiting for operations in transaction {} to complete before committing", (Object)this.id);
        this.operationPhaser.register();
        this.operationPhaser.awaitAdvance(this.operationPhaser.arriveAndDeregister());
        log.debug("Committing transaction {}", (Object)this.id);
        try {
            if (this.isShortLived()) {
                this.doCommitShortLived();
            } else {
                this.doCommitLongRunning();
            }
            this.updateState(TransactionState.COMMITTED);
            this.getEventAccumulator().emitEvents((Transaction)this, this.baseUri, this.userAgent);
            this.releaseLocks();
            log.debug("Committed transaction {}", (Object)this.id);
        }
        catch (Exception ex) {
            log.error("Failed to commit transaction: {}", (Object)this.id, (Object)ex);
            log.info("Rolling back transaction {}", (Object)this.id);
            this.rollback();
            throw new RepositoryRuntimeException("Failed to commit transaction " + this.id, (Throwable)ex);
        }
    }

    public boolean isCommitted() {
        return this.state == TransactionState.COMMITTED;
    }

    public synchronized void rollback() {
        if (this.state == TransactionState.ROLLEDBACK || this.state == TransactionState.ROLLINGBACK) {
            return;
        }
        this.failIfCommitted();
        this.updateState(TransactionState.ROLLINGBACK);
        log.debug("Waiting for operations in transaction {} to complete before rolling back", (Object)this.id);
        this.operationPhaser.register();
        this.operationPhaser.awaitAdvance(this.operationPhaser.arriveAndDeregister());
        this.execQuietly("Failed to rollback storage in transaction " + this.id, () -> this.getPersistentSession().rollback());
        this.execQuietly("Failed to rollback index in transaction " + this.id, () -> this.getContainmentIndex().rollbackTransaction((Transaction)this));
        this.execQuietly("Failed to rollback reference index in transaction " + this.id, () -> this.getReferenceService().rollbackTransaction((Transaction)this));
        this.execQuietly("Failed to rollback membership index in transaction " + this.id, () -> this.getMembershipService().rollbackTransaction((Transaction)this));
        this.execQuietly("Failed to rollback search index in transaction " + this.id, () -> this.getSearchIndex().rollbackTransaction((Transaction)this));
        this.execQuietly("Failed to rollback events in transaction " + this.id, () -> this.getEventAccumulator().clearEvents((Transaction)this));
        this.execQuietly("Failed to clear user rdf types cache in transaction " + this.id, () -> this.getUserTypesCache().dropSessionCache(this.id));
        this.updateState(TransactionState.ROLLEDBACK);
        this.releaseLocks();
    }

    public void doInTx(Runnable runnable) {
        this.operationPhaser.register();
        try {
            this.failIfNotOpen();
            this.failIfExpired();
            runnable.run();
        }
        finally {
            this.operationPhaser.arriveAndDeregister();
        }
    }

    public synchronized void fail() {
        if (this.state != TransactionState.OPEN) {
            log.error("Transaction {} is in state {} and may not be marked as FAILED", (Object)this.id, (Object)this.state);
        } else {
            this.updateState(TransactionState.FAILED);
        }
    }

    public boolean isRolledBack() {
        return this.state == TransactionState.ROLLEDBACK;
    }

    public String getId() {
        return this.id;
    }

    public void setShortLived(boolean shortLived) {
        this.shortLived = shortLived;
    }

    public boolean isShortLived() {
        return this.shortLived;
    }

    public boolean isOpenLongRunning() {
        return !this.isShortLived() && !this.hasExpired() && this.state != TransactionState.COMMITTED && this.state != TransactionState.ROLLEDBACK && this.state != TransactionState.FAILED;
    }

    public boolean isOpen() {
        return this.state == TransactionState.OPEN && !this.hasExpired();
    }

    public void ensureCommitting() {
        if (this.state != TransactionState.COMMITTING) {
            throw new TransactionRuntimeException(String.format("Transaction %s must be in state COMMITTING, but was %s", this.id, this.state));
        }
    }

    public boolean isReadOnly() {
        return false;
    }

    public void expire() {
        this.expiration = Instant.now();
        this.expired = true;
    }

    public boolean hasExpired() {
        if (this.expired) {
            return true;
        }
        this.expired = this.expiration.isBefore(Instant.now());
        return this.expired;
    }

    public synchronized Instant updateExpiry(Duration amountToAdd) {
        this.failIfExpired();
        this.failIfCommitted();
        this.failIfNotOpen();
        this.expiration = this.expiration.plus(amountToAdd);
        return this.expiration;
    }

    public Instant getExpires() {
        return this.expiration;
    }

    public void commitIfShortLived() {
        if (this.isShortLived()) {
            this.commit();
        }
    }

    public void refresh() {
        this.updateExpiry(this.sessionTimeout);
    }

    public void lockResource(FedoraId resourceId) {
        this.getResourceLockManger().acquire(this.getId(), resourceId);
    }

    public void releaseResourceLocksIfShortLived() {
        if (this.isShortLived()) {
            this.releaseLocks();
        }
    }

    public void setBaseUri(String baseUri) {
        this.baseUri = baseUri;
    }

    public void setUserAgent(String userAgent) {
        this.userAgent = userAgent;
    }

    private void doCommitShortLived() {
        this.getPersistentSession().prepare();
        this.getPersistentSession().commit();
        this.getUserTypesCache().mergeSessionCache(this.id);
    }

    private void doCommitLongRunning() {
        this.getDbTransactionExecutor().doInTxWithRetry(() -> {
            this.getContainmentIndex().commitTransaction((Transaction)this);
            this.getReferenceService().commitTransaction((Transaction)this);
            this.getMembershipService().commitTransaction((Transaction)this);
            this.getSearchIndex().commitTransaction((Transaction)this);
            this.getPersistentSession().prepare();
            this.getPersistentSession().commit();
            this.getUserTypesCache().mergeSessionCache(this.id);
        });
    }

    private void updateState(TransactionState newState) {
        this.state = newState;
    }

    private PersistentStorageSession getPersistentSession() {
        return this.txManager.getPersistentStorageSessionManager().getSession((Transaction)this);
    }

    private void failIfExpired() {
        if (this.hasExpired()) {
            throw new TransactionClosedException("Transaction " + this.id + " expired!");
        }
    }

    private void failIfCommitted() {
        if (this.state == TransactionState.COMMITTED) {
            throw new TransactionClosedException(String.format("Transaction %s cannot be transitioned because it is already committed!", this.id));
        }
    }

    private void failIfNotOpen() {
        if (this.state == TransactionState.FAILED) {
            throw new TransactionRuntimeException(String.format("Transaction %s cannot be committed because it is in a failed state!", this.id));
        }
        if (this.state != TransactionState.OPEN) {
            throw new TransactionClosedException(String.format("Transaction %s cannot be committed because it is in state %s!", this.id, this.state));
        }
    }

    private void releaseLocks() {
        this.execQuietly("Failed to release resource locks cleanly. You may need to restart Fedora.", () -> this.getResourceLockManger().releaseAll(this.getId()));
    }

    private void execQuietly(String failureMessage, CheckedRunnable callable) {
        try {
            callable.run();
        }
        catch (Exception e) {
            log.error(failureMessage, (Throwable)e);
        }
    }

    private ContainmentIndex getContainmentIndex() {
        return this.txManager.getContainmentIndex();
    }

    private EventAccumulator getEventAccumulator() {
        return this.txManager.getEventAccumulator();
    }

    private ReferenceService getReferenceService() {
        return this.txManager.getReferenceService();
    }

    private MembershipService getMembershipService() {
        return this.txManager.getMembershipService();
    }

    private SearchIndex getSearchIndex() {
        return this.txManager.getSearchIndex();
    }

    private DbTransactionExecutor getDbTransactionExecutor() {
        return this.txManager.getDbTransactionExecutor();
    }

    private ResourceLockManager getResourceLockManger() {
        return this.txManager.getResourceLockManager();
    }

    private UserTypesCache getUserTypesCache() {
        return this.txManager.getUserTypesCache();
    }

    public String toString() {
        return this.id;
    }
}

