/**
 * Copyright 2013 DuraSpace, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.fcrepo.kernel.rdf.impl;

import static com.google.common.base.Predicates.not;
import static com.hp.hpl.jena.graph.Triple.create;
import static com.hp.hpl.jena.rdf.model.ResourceFactory.createResource;
import static org.fcrepo.jcr.FedoraJcrTypes.ROOT;
import static org.fcrepo.kernel.RdfLexicon.HAS_CONTENT;
import static org.fcrepo.kernel.RdfLexicon.HAS_CONTENT_LOCATION;
import static org.fcrepo.kernel.RdfLexicon.IS_CONTENT_OF;
import static org.fcrepo.kernel.utils.FedoraTypesUtils.isBinaryContentProperty;
import static org.fcrepo.kernel.utils.FedoraTypesUtils.property2values;
import static org.modeshape.jcr.api.JcrConstants.JCR_CONTENT;
import static org.slf4j.LoggerFactory.getLogger;

import java.util.Iterator;

import javax.jcr.AccessDeniedException;
import javax.jcr.Property;
import javax.jcr.RepositoryException;

import org.fcrepo.kernel.rdf.IdentifierTranslator;
import org.fcrepo.kernel.rdf.impl.mappings.PropertyToTriple;
import org.fcrepo.kernel.rdf.impl.mappings.ZippingIterator;
import org.fcrepo.kernel.services.LowLevelStorageService;
import org.fcrepo.kernel.utils.LowLevelCacheEntry;
import org.fcrepo.kernel.utils.iterators.PropertyIterator;
import org.slf4j.Logger;

import com.google.common.base.Function;
import com.google.common.collect.Iterators;
import com.google.common.collect.UnmodifiableIterator;
import com.hp.hpl.jena.graph.Node;
import com.hp.hpl.jena.graph.Triple;

/**
 * {@link NodeRdfContext} for RDF that derives from JCR properties on a
 * {@link Node}.
 *
 * @author ajs6f
 * @date Oct 10, 2013
 */
public class PropertiesRdfContext extends NodeRdfContext {

    private PropertyToTriple property2triple;

    private static final Logger LOGGER = getLogger(PropertiesRdfContext.class);

    /**
     * Default constructor.
     *
     * @param node
     * @throws RepositoryException
     */

    public PropertiesRdfContext(final javax.jcr.Node node, final IdentifierTranslator graphSubjects,
        final LowLevelStorageService lowLevelStorageService) throws RepositoryException {
        super(node, graphSubjects, lowLevelStorageService);
        property2triple = new PropertyToTriple(graphSubjects);
        putPropertiesIntoContext();
    }

    private void putPropertiesIntoContext() throws RepositoryException {

        LOGGER.trace(
                "Pushing RDF triples into context for properties of node: {}",
                node());

        // this node's own properties
        if (node().hasProperties()) {
            concat(triplesFromProperties(node()));
        }

        // if there's an accessible jcr:content node, include information about
        // it
        javax.jcr.Node contentNode = null;
        try {
            if (node().hasNode(JCR_CONTENT)) {
                contentNode = node().getNode(JCR_CONTENT);
            }
        } catch (final AccessDeniedException e) {
            LOGGER.trace("Access denied to content node", e);
        }
        if (contentNode != null) {
            final Node contentSubject = graphSubjects().getSubject(contentNode.getPath()).asNode();
            final Node subject = graphSubjects().getSubject(node().getPath()).asNode();
            // add triples representing parent-to-content-child relationship
            concat(new Triple[] {
                    create(subject, HAS_CONTENT.asNode(), contentSubject),
                    create(contentSubject, IS_CONTENT_OF.asNode(), subject)});
            // add properties from content child
            concat(new PropertiesRdfContext(node().getNode(JCR_CONTENT),
                    graphSubjects(), lowLevelStorageService()));

            // add triples describing storage of content child
            lowLevelStorageService().setRepository(
                    node().getSession().getRepository());
            concat(Iterators.transform(lowLevelStorageService().getLowLevelCacheEntries(
                    contentNode).iterator(),
                    new Function<LowLevelCacheEntry, Triple>() {

                        @Override
                        public Triple apply(final LowLevelCacheEntry llce) {
                            return create(contentSubject,
                                    HAS_CONTENT_LOCATION.asNode(),
                                    createResource(llce.getExternalIdentifier()).asNode());
                        }
                    }));

        }

        if (node().getPrimaryNodeType().getName().equals(ROOT)) {
            concat(new RootRdfContext(node(), graphSubjects(), lowLevelStorageService()));
        }

    }

    private Iterator<Triple> triplesFromProperties(final javax.jcr.Node n)
        throws RepositoryException {
        LOGGER.trace("Creating triples for node: {}", n);
        final UnmodifiableIterator<Property> nonBinaryProperties =
            Iterators.filter(new PropertyIterator(n.getProperties()),
                    not(isBinaryContentProperty));

        final UnmodifiableIterator<Property> nonBinaryPropertiesCopy =
            Iterators.filter(new PropertyIterator(n.getProperties()),
                    not(isBinaryContentProperty));

        return Iterators.concat(new ZippingIterator<>(
                Iterators.transform(
                    nonBinaryProperties, property2values),
                Iterators.transform(
                    nonBinaryPropertiesCopy, property2triple)));

    }

}
