/*
 * Decompiled with CFR 0.152.
 */
package edu.wisc.library.ocfl.core.validation;

import edu.wisc.library.ocfl.api.exception.CorruptObjectException;
import edu.wisc.library.ocfl.api.exception.InvalidInventoryException;
import edu.wisc.library.ocfl.api.exception.OcflIOException;
import edu.wisc.library.ocfl.api.model.DigestAlgorithm;
import edu.wisc.library.ocfl.api.model.OcflVersion;
import edu.wisc.library.ocfl.api.model.VersionNum;
import edu.wisc.library.ocfl.api.util.Enforce;
import edu.wisc.library.ocfl.core.ObjectPaths;
import edu.wisc.library.ocfl.core.inventory.InventoryMapper;
import edu.wisc.library.ocfl.core.inventory.SidecarMapper;
import edu.wisc.library.ocfl.core.model.Inventory;
import edu.wisc.library.ocfl.core.model.Version;
import edu.wisc.library.ocfl.core.util.DigestUtil;
import edu.wisc.library.ocfl.core.util.FileUtil;
import edu.wisc.library.ocfl.core.util.NamasteTypeFile;
import edu.wisc.library.ocfl.core.validation.InventoryValidator;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.FileVisitOption;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

public class ObjectValidator {
    private final InventoryMapper inventoryMapper;

    public ObjectValidator(InventoryMapper inventoryMapper) {
        this.inventoryMapper = Enforce.notNull(inventoryMapper, "inventoryMapper cannot be null");
    }

    public void validateObject(Path objectRoot, Inventory inventory) {
        InventoryValidator.validateDeep(inventory);
        this.validateObjectStructure(objectRoot, inventory);
        this.validateRootInventorySameAsHeadInventory(objectRoot, inventory);
        this.validateManifest(objectRoot, inventory);
        this.validateVersions(objectRoot, inventory);
    }

    public void validateVersion(Path versionRoot, Inventory inventory) {
        this.validateInventory(versionRoot);
        this.validateVersionRoot(versionRoot);
        this.validateVersionFiles(versionRoot, inventory);
    }

    private void validateObjectStructure(Path objectRoot, Inventory inventory) {
        OcflVersion ocflVersion = inventory.getType().getOcflVersion();
        HashSet<Path> expectedFiles = new HashSet<Path>();
        expectedFiles.add(this.validateNamasteFile(ocflVersion, objectRoot));
        this.validateInventory(objectRoot);
        expectedFiles.add(ObjectPaths.inventoryPath(objectRoot));
        expectedFiles.add(ObjectPaths.inventorySidecarPath(objectRoot, inventory));
        expectedFiles.add(ObjectPaths.logsPath(objectRoot));
        expectedFiles.add(ObjectPaths.extensionsPath(objectRoot));
        inventory.getVersions().keySet().forEach(version2 -> {
            if (!inventory.hasMutableHead() || !version2.equals(inventory.getHead())) {
                Path versionDir = this.validatePathExists(objectRoot.resolve(version2.toString()));
                this.validateInventory(versionDir);
                this.validateVersionRoot(versionDir);
                expectedFiles.add(versionDir);
            }
        });
        try (Stream<Path> walk = Files.list(objectRoot);){
            walk.filter(file -> !expectedFiles.contains(file)).forEach(file -> {
                throw new CorruptObjectException(String.format("Object %s contains an invalid file: %s", inventory.getId(), file));
            });
        }
        catch (IOException e2) {
            throw new OcflIOException(e2);
        }
    }

    private void validateManifest(final Path objectRoot, final Inventory inventory) {
        HashMap<String, Set<String>> manifestCopy;
        block9: {
            final DigestAlgorithm algorithm = inventory.getDigestAlgorithm();
            final HashSet<Path> excludeDirs = new HashSet<Path>();
            excludeDirs.add(ObjectPaths.extensionsPath(objectRoot));
            excludeDirs.add(ObjectPaths.logsPath(objectRoot));
            manifestCopy = new HashMap<String, Set<String>>(inventory.getManifest());
            try {
                Path mutableContentDir;
                Files.walkFileTree(objectRoot, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

                    @Override
                    public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) {
                        VersionNum version2;
                        if (excludeDirs.contains(dir)) {
                            return FileVisitResult.SKIP_SUBTREE;
                        }
                        if (dir.getParent().equals(objectRoot) && (version2 = VersionNum.fromString(dir.getFileName().toString())).compareTo(inventory.getHead()) > 0) {
                            return FileVisitResult.SKIP_SUBTREE;
                        }
                        return FileVisitResult.CONTINUE;
                    }

                    @Override
                    public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) {
                        if (!file.getParent().equals(objectRoot) && !file.getParent().getParent().equals(objectRoot)) {
                            String digest = DigestUtil.computeDigestHex(algorithm, file);
                            String contentPath = ObjectValidator.this.validateManifestFile(inventory, objectRoot, file, digest);
                            if (contentPath != null) {
                                if (((Set)manifestCopy.get(digest)).size() == 1) {
                                    manifestCopy.remove(digest);
                                } else {
                                    ((Set)manifestCopy.get(digest)).remove(contentPath);
                                }
                                return FileVisitResult.CONTINUE;
                            }
                            throw new CorruptObjectException(String.format("File %s has unexpected %s digest value %s.", file, algorithm.getOcflName(), digest));
                        }
                        return FileVisitResult.CONTINUE;
                    }
                });
                if (!inventory.hasMutableHead() || !Files.exists(mutableContentDir = ObjectPaths.mutableHeadVersionPath(objectRoot).resolve(inventory.resolveContentDirectory()), new LinkOption[0])) break block9;
                try (Stream<Path> files = Files.walk(mutableContentDir, new FileVisitOption[0]);){
                    files.filter(x$0 -> Files.isRegularFile(x$0, new LinkOption[0])).forEach(file -> {
                        String digest = DigestUtil.computeDigestHex(algorithm, file);
                        String contentPath = this.validateManifestFile(inventory, objectRoot, (Path)file, digest);
                        if (contentPath != null) {
                            if (((Set)manifestCopy.get(digest)).size() == 1) {
                                manifestCopy.remove(digest);
                            } else {
                                ((Set)manifestCopy.get(digest)).remove(contentPath);
                            }
                        } else {
                            throw new CorruptObjectException(String.format("File %s has unexpected %s digest value %s.", file, algorithm.getOcflName(), digest));
                        }
                    });
                }
            }
            catch (IOException e2) {
                throw new OcflIOException(e2);
            }
        }
        if (!manifestCopy.isEmpty()) {
            throw new CorruptObjectException(String.format("The following files are defined in object %s's manifest, but are not found on disk: %s", inventory.getId(), manifestCopy));
        }
    }

    private String validateManifestFile(Inventory inventory, Path objectRoot, Path file, String digest) {
        Set<String> contentPaths = inventory.getContentPaths(digest);
        if (contentPaths.isEmpty()) {
            throw new CorruptObjectException(String.format("Object %s contains an unexpected file: %s", inventory.getId(), file));
        }
        for (String contentPath : contentPaths) {
            if (!file.equals(objectRoot.resolve(contentPath))) continue;
            return contentPath;
        }
        return null;
    }

    private void validateVersionFiles(Path versionRoot, Inventory inventory) {
        Path contentDirectory = versionRoot.resolve(inventory.resolveContentDirectory());
        String contentPathPrefix = FileUtil.pathJoinFailEmpty(inventory.getHead().toString(), inventory.resolveContentDirectory());
        Set<String> expectedFileIds = inventory.getFileIdsForMatchingFiles(contentPathPrefix);
        HashMap expectedManifest = new HashMap(expectedFileIds.size());
        for (String fileId : expectedFileIds) {
            Set filteredContentPaths = inventory.getContentPaths(fileId).stream().filter(path -> path.startsWith(contentPathPrefix)).collect(Collectors.toSet());
            expectedManifest.put(fileId, filteredContentPaths);
        }
        if (Files.exists(contentDirectory, new LinkOption[0])) {
            try (Stream<Path> files = Files.walk(contentDirectory, new FileVisitOption[0]);){
                files.filter(x$0 -> Files.isRegularFile(x$0, new LinkOption[0])).forEach(file -> {
                    Path contentRelative = contentDirectory.relativize((Path)file);
                    String contentPath = FileUtil.pathJoinFailEmpty(contentPathPrefix, FileUtil.pathToStringStandardSeparator(contentRelative));
                    String expectedDigest = inventory.getFileId(contentPath);
                    if (expectedDigest == null) {
                        throw new CorruptObjectException(String.format("Object %s contains an unexpected file: %s", inventory.getId(), file));
                    }
                    String digest = DigestUtil.computeDigestHex(inventory.getDigestAlgorithm(), file);
                    if (!expectedDigest.equalsIgnoreCase(digest)) {
                        throw new CorruptObjectException(String.format("File %s has unexpected %s digest value. Expected: %s; Actual: %s.", file, inventory.getDigestAlgorithm().getOcflName(), expectedDigest, digest));
                    }
                    if (((Set)expectedManifest.get(digest)).size() == 1) {
                        expectedManifest.remove(digest);
                    } else {
                        ((Set)expectedManifest.get(digest)).remove(contentPath);
                    }
                });
            }
            catch (IOException e2) {
                throw new OcflIOException(e2);
            }
        }
        if (!expectedManifest.isEmpty()) {
            throw new CorruptObjectException(String.format("The following files are defined in object %s's manifest, but are not found on disk: %s", inventory.getId(), expectedManifest));
        }
    }

    private void validateVersions(Path objectRoot, Inventory inventory) {
        Inventory currentInventory = inventory;
        while (!VersionNum.V1.equals(currentInventory.getHead())) {
            VersionNum previous = currentInventory.getHead().previousVersionNum();
            Path inventoryPath = ObjectPaths.inventoryPath(objectRoot.resolve(previous.toString()));
            Inventory previousInventory = this.inventoryMapper.read(inventory.getObjectRootPath(), "digest", inventoryPath);
            InventoryValidator.validateDeep(previousInventory);
            this.validateId(inventory, previousInventory);
            this.validateVersionNumber(previous, previousInventory, inventoryPath);
            if (!Objects.equals(currentInventory.getDigestAlgorithm(), previousInventory.getDigestAlgorithm())) {
                this.validateManifest(objectRoot, previousInventory);
                this.validateVersionStatesByContentPath(currentInventory, previousInventory);
            } else {
                InventoryValidator.validateVersionStates(currentInventory, previousInventory);
            }
            currentInventory = previousInventory;
        }
    }

    private Path validateNamasteFile(OcflVersion ocflVersion, Path objectRoot) {
        NamasteTypeFile namasteFile = new NamasteTypeFile(ocflVersion.getOcflObjectVersion());
        Path namasteFilePath = this.validatePathExists(objectRoot.resolve(namasteFile.fileName()));
        String actualContent = this.content(namasteFilePath);
        if (Objects.equals(namasteFile.fileContent(), actualContent.trim())) {
            throw new CorruptObjectException("Invalid namaste file at " + namasteFilePath);
        }
        return namasteFilePath;
    }

    private void validateInventory(Path path) {
        Path inventory = this.validatePathExists(ObjectPaths.inventoryPath(path));
        Path sidecar = this.validatePathExists(ObjectPaths.findInventorySidecarPath(path));
        DigestAlgorithm algorithm = SidecarMapper.getDigestAlgorithmFromSidecar(FileUtil.pathToStringStandardSeparator(sidecar));
        this.validateInventoryDigest(inventory, sidecar, algorithm);
    }

    private void validateInventoryDigest(Path inventory, Path sidecar, DigestAlgorithm algorithm) {
        String actual;
        String expected = SidecarMapper.readDigest(sidecar);
        if (!expected.equalsIgnoreCase(actual = DigestUtil.computeDigestHex(algorithm, inventory))) {
            throw new CorruptObjectException(String.format("Inventory file at %s does not match expected %s digest: Expected %s; Actual %s", inventory, algorithm.getOcflName(), expected, actual));
        }
    }

    private void validateVersionRoot(Path versionRoot) {
        String inventory = ObjectPaths.inventoryPath(versionRoot).getFileName().toString();
        String sidecar = ObjectPaths.findInventorySidecarPath(versionRoot).getFileName().toString();
        try (Stream<Path> files = Files.list(versionRoot);){
            files.filter(x$0 -> Files.isRegularFile(x$0, new LinkOption[0])).forEach(file -> {
                String name = file.getFileName().toString();
                if (!name.equals(inventory) && !name.equals(sidecar)) {
                    throw new CorruptObjectException("Version contains an illegal file at " + file);
                }
            });
        }
        catch (IOException e2) {
            throw new OcflIOException(e2);
        }
    }

    private void validateVersionStatesByContentPath(Inventory currentInventory, Inventory previousInventory) {
        VersionNum current = previousInventory.getHead();
        while (true) {
            VersionNum currentCopy = current;
            Version currentVersion = currentInventory.getVersion(current);
            Map<String, Set<String>> currentState = currentVersion.getState();
            Map<String, Set<String>> previousState = previousInventory.getVersion(current).getState();
            previousState.values().stream().flatMap(Collection::stream).forEach(previousLogical -> {
                if (currentVersion.getFileId((String)previousLogical) == null) {
                    throw this.versionMismatchException(currentInventory, previousInventory, currentCopy);
                }
            });
            currentState.forEach((currentDigest, currentLogical) -> {
                Set<String> currentContentPaths = currentInventory.getContentPaths((String)currentDigest);
                for (String currentContentPath : currentContentPaths) {
                    String previousDigest = previousInventory.getFileId(currentContentPath);
                    Set previousLogical = (Set)previousState.get(previousDigest);
                    if (Objects.equals(currentLogical, previousLogical)) continue;
                    throw this.versionMismatchException(currentInventory, previousInventory, currentCopy);
                }
            });
            if (VersionNum.V1.equals(current)) break;
            current = current.previousVersionNum();
        }
    }

    private Path validatePathExists(Path path) {
        if (!Files.exists(path, new LinkOption[0])) {
            throw new CorruptObjectException(String.format("Expected file %s to exist, but it does not.", path));
        }
        return path;
    }

    private void validateRootInventorySameAsHeadInventory(Path objectRoot, Inventory inventory) {
        String headDigest;
        String rootDigest;
        VersionNum rootVersion = inventory.getHead();
        if (inventory.hasMutableHead()) {
            rootVersion = rootVersion.previousVersionNum();
        }
        if (!(rootDigest = this.content(ObjectPaths.findInventorySidecarPath(objectRoot))).equalsIgnoreCase(headDigest = this.content(ObjectPaths.findInventorySidecarPath(objectRoot.resolve(rootVersion.toString()))))) {
            throw new CorruptObjectException(String.format("The inventory file in the object root of object %s does not match the inventory in the version directory %s", inventory.getId(), rootVersion));
        }
    }

    private void validateId(Inventory currentInventory, Inventory previousInventory) {
        if (!Objects.equals(currentInventory.getId(), previousInventory.getId())) {
            throw new CorruptObjectException(String.format("Versions %s and %s of object %s have different object IDs, %s and %s.", currentInventory.getHead(), previousInventory.getHead(), currentInventory.getId(), currentInventory.getId(), previousInventory.getId()));
        }
    }

    private void validateVersionNumber(VersionNum expectedId, Inventory inventory, Path inventoryPath) {
        if (!Objects.equals(expectedId.toString(), inventory.getHead().toString())) {
            throw new CorruptObjectException(String.format("Expected version %s but was %s in %s.", expectedId, inventory.getHead(), inventoryPath));
        }
    }

    private String content(Path file) {
        try {
            return Files.readString(file, StandardCharsets.UTF_8);
        }
        catch (IOException e2) {
            throw new OcflIOException(e2);
        }
    }

    private InvalidInventoryException versionMismatchException(Inventory currentInventory, Inventory previousInventory, VersionNum current) {
        return new InvalidInventoryException(String.format("In object %s the inventories in version %s and %s define a different state for version %s.", currentInventory.getId(), currentInventory.getHead(), previousInventory.getHead(), current));
    }
}

