/*
 * Decompiled with CFR 0.152.
 */
package edu.wisc.library.ocfl.core.util;

import edu.wisc.library.ocfl.api.OcflOption;
import edu.wisc.library.ocfl.api.exception.OcflIOException;
import edu.wisc.library.ocfl.api.model.DigestAlgorithm;
import edu.wisc.library.ocfl.api.util.Enforce;
import edu.wisc.library.ocfl.core.util.DigestUtil;
import edu.wisc.library.ocfl.core.util.UncheckedFiles;
import java.io.IOException;
import java.nio.file.AtomicMoveNotSupportedException;
import java.nio.file.CopyOption;
import java.nio.file.DirectoryStream;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.FileVisitOption;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.security.SecureRandom;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Stream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class FileUtil {
    private static final Logger LOG = LoggerFactory.getLogger(FileUtil.class);
    private static final SecureRandom RANDOM = new SecureRandom();

    private FileUtil() {
    }

    public static Path createObjectTempDir(Path parent, String objectId) {
        String digest = DigestUtil.computeDigestHex(DigestAlgorithm.md5, objectId);
        UncheckedFiles.createDirectories(parent);
        return UncheckedFiles.createDirectory(parent.resolve(digest + "-" + Long.toUnsignedString(RANDOM.nextLong())));
    }

    public static void moveDirectory(final Path srcRoot, final Path dstRoot) throws FileAlreadyExistsException {
        if (Files.notExists(srcRoot, new LinkOption[0]) || Files.isRegularFile(srcRoot, new LinkOption[0])) {
            throw new IllegalArgumentException("Source must exist and be a directory: " + srcRoot);
        }
        if (Files.exists(dstRoot, new LinkOption[0])) {
            throw new FileAlreadyExistsException("Destination must not exist: " + dstRoot);
        }
        if (Files.notExists(dstRoot.getParent(), new LinkOption[0]) || Files.isRegularFile(dstRoot.getParent(), new LinkOption[0])) {
            throw new IllegalArgumentException("Parent directory of destination must exist and be a directory: " + dstRoot.getParent());
        }
        try {
            Files.move(srcRoot, dstRoot, StandardCopyOption.ATOMIC_MOVE);
            return;
        }
        catch (AtomicMoveNotSupportedException e2) {
            LOG.debug("Atomic move of {} to {} failed.", srcRoot, dstRoot, e2);
        }
        catch (FileAlreadyExistsException e3) {
            throw e3;
        }
        catch (IOException e4) {
            throw new OcflIOException(e4);
        }
        try {
            Files.createDirectory(dstRoot, new FileAttribute[0]);
        }
        catch (FileAlreadyExistsException e5) {
            throw e5;
        }
        catch (IOException e6) {
            throw new OcflIOException(e6);
        }
        try {
            Files.walkFileTree(srcRoot, Set.of(FileVisitOption.FOLLOW_LINKS), Integer.MAX_VALUE, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

                private Path dstPath(Path current) {
                    return dstRoot.resolve(srcRoot.relativize(current));
                }

                @Override
                public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
                    if (!dir.equals(srcRoot)) {
                        Files.createDirectories(this.dstPath(dir), new FileAttribute[0]);
                    }
                    return super.preVisitDirectory(dir, attrs);
                }

                @Override
                public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                    Files.move(file, this.dstPath(file), StandardCopyOption.REPLACE_EXISTING);
                    return super.visitFile(file, attrs);
                }

                @Override
                public FileVisitResult postVisitDirectory(Path dir, IOException exc) throws IOException {
                    Files.delete(dir);
                    return super.postVisitDirectory(dir, exc);
                }
            });
        }
        catch (IOException e7) {
            throw new OcflIOException(e7);
        }
    }

    public static void recursiveCopy(final Path src, final Path dst, final StandardCopyOption ... copyOptions) {
        try {
            Files.createDirectories(dst, new FileAttribute[0]);
            Files.walkFileTree(src, Set.of(FileVisitOption.FOLLOW_LINKS), Integer.MAX_VALUE, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

                private Path dstPath(Path current) {
                    return dst.resolve(src.relativize(current));
                }

                @Override
                public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
                    if (!dir.equals(src)) {
                        Files.createDirectories(this.dstPath(dir), new FileAttribute[0]);
                    }
                    return super.preVisitDirectory(dir, attrs);
                }

                @Override
                public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                    Files.copy(file, this.dstPath(file), (CopyOption[])copyOptions);
                    return super.visitFile(file, attrs);
                }
            });
        }
        catch (IOException e2) {
            throw new OcflIOException(e2);
        }
    }

    public static void copyFileMakeParents(Path src, Path dst, StandardCopyOption ... copyOptions) {
        try {
            Files.createDirectories(dst.getParent(), new FileAttribute[0]);
            Files.copy(src, dst, (CopyOption[])copyOptions);
        }
        catch (IOException e2) {
            throw new OcflIOException(e2);
        }
    }

    public static void moveFileMakeParents(Path src, Path dst, StandardCopyOption ... copyOptions) {
        try {
            Files.createDirectories(dst.getParent(), new FileAttribute[0]);
            Files.move(src, dst, copyOptions);
        }
        catch (IOException e2) {
            throw new OcflIOException(e2);
        }
    }

    public static void deleteChildren(Path root) {
        try (Stream<Path> files = Files.walk(root, new FileVisitOption[0]);){
            files.sorted(Comparator.reverseOrder()).filter(f -> !f.equals(root)).forEach(UncheckedFiles::delete);
        }
        catch (IOException e2) {
            throw new OcflIOException(e2);
        }
    }

    public static void deleteEmptyDirs(Path root) {
        try (Stream<Path> files = Files.walk(root, new FileVisitOption[0]);){
            files.filter(f -> Files.isDirectory(f, LinkOption.NOFOLLOW_LINKS)).filter(f -> !f.equals(root)).filter(FileUtil::isDirEmpty).forEach(UncheckedFiles::delete);
        }
        catch (IOException e2) {
            throw new OcflIOException(e2);
        }
    }

    public static void deleteDirAndParentsIfEmpty(Path path) {
        if (FileUtil.isDirEmpty(path)) {
            UncheckedFiles.delete(path);
            FileUtil.deleteDirAndParentsIfEmpty(path.getParent());
        }
    }

    public static void safeDeleteDirectory(Path path) {
        if (Files.exists(path, new LinkOption[0])) {
            try (Stream<Path> paths = Files.walk(path, new FileVisitOption[0]);){
                paths.sorted(Comparator.reverseOrder()).forEach(f -> {
                    try {
                        Files.delete(f);
                    }
                    catch (NoSuchFileException noSuchFileException) {
                    }
                    catch (IOException e2) {
                        LOG.warn("Failed to delete file: {}", f, (Object)e2);
                    }
                });
            }
            catch (IOException e2) {
                LOG.warn("Failed to delete directory: {}", (Object)path, (Object)e2);
            }
        }
    }

    public static void deleteDirectory(Path directory) {
        AtomicBoolean hasErrors = new AtomicBoolean(false);
        if (Files.exists(directory, new LinkOption[0])) {
            try (Stream<Path> paths = Files.walk(directory, new FileVisitOption[0]);){
                paths.sorted(Comparator.reverseOrder()).forEach(f -> {
                    try {
                        Files.delete(f);
                    }
                    catch (IOException e2) {
                        LOG.warn("Failed to delete file: {}", f, (Object)e2);
                        hasErrors.set(true);
                    }
                });
            }
            catch (IOException e2) {
                throw new OcflIOException(e2);
            }
        }
        if (hasErrors.get()) {
            throw new RuntimeException(String.format("Failed to recursively delete directory %s. See logs for details.", directory));
        }
    }

    public static boolean isDirEmpty(Path path) {
        boolean bl;
        block8: {
            DirectoryStream<Path> stream = Files.newDirectoryStream(path);
            try {
                boolean bl2 = bl = !stream.iterator().hasNext();
                if (stream == null) break block8;
            }
            catch (Throwable throwable) {
                try {
                    if (stream != null) {
                        try {
                            stream.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException e2) {
                    throw new OcflIOException(e2);
                }
            }
            stream.close();
        }
        return bl;
    }

    public static List<Path> findFiles(Path path) {
        ArrayList<Path> files;
        block9: {
            files = new ArrayList<Path>();
            if (Files.isDirectory(path, LinkOption.NOFOLLOW_LINKS)) {
                try (Stream<Path> paths = Files.walk(path, new FileVisitOption[0]);){
                    paths.filter(x$0 -> Files.isRegularFile(x$0, new LinkOption[0])).forEach(files::add);
                    break block9;
                }
                catch (IOException e2) {
                    throw new OcflIOException(e2);
                }
            }
            files.add(path);
        }
        return files;
    }

    public static StandardCopyOption[] toCopyOptions(OcflOption ... options) {
        if (OcflOption.contains(OcflOption.OVERWRITE, options)) {
            return new StandardCopyOption[]{StandardCopyOption.REPLACE_EXISTING};
        }
        return new StandardCopyOption[0];
    }

    public static boolean hasChildren(Path path) {
        if (Files.exists(path, new LinkOption[0]) && Files.isDirectory(path, new LinkOption[0])) {
            return !FileUtil.isDirEmpty(path);
        }
        return false;
    }

    public static String pathToStringStandardSeparator(Path path) {
        Enforce.notNull(path, "path cannot be null");
        char separator = path.getFileSystem().getSeparator().charAt(0);
        String pathStr = path.toString();
        if (separator == '/') {
            return pathStr;
        }
        return pathStr.replace(separator, '/');
    }

    public static String pathJoinIgnoreEmpty(String ... parts) {
        return FileUtil.pathJoin(false, parts);
    }

    public static String pathJoinFailEmpty(String ... parts) {
        return FileUtil.pathJoin(true, parts);
    }

    private static String pathJoin(boolean failOnEmpty, String ... parts) {
        if (parts == null || parts.length == 0) {
            return "";
        }
        StringBuilder pathBuilder = new StringBuilder();
        boolean addSeparator = false;
        for (int i = 0; i < parts.length; ++i) {
            String part = parts[i];
            if (failOnEmpty && (part == null || part.isEmpty())) {
                throw new IllegalArgumentException(String.format("Path cannot be joined because it contains empty parts: %s", Arrays.asList(parts)));
            }
            if (part == null || part.isEmpty()) continue;
            String strippedPart = i == 0 ? FileUtil.firstPathPart(part) : FileUtil.stripSlashes(part);
            if (!strippedPart.isEmpty()) {
                if (addSeparator) {
                    pathBuilder.append("/");
                }
                pathBuilder.append(strippedPart);
                addSeparator = true;
                continue;
            }
            if (!failOnEmpty) continue;
            throw new IllegalArgumentException(String.format("Path cannot be joined because it contains empty parts: %s", Arrays.asList(parts)));
        }
        return pathBuilder.toString();
    }

    private static String stripSlashes(String path) {
        int startIndex;
        int endIndex = path.length();
        for (startIndex = 0; startIndex < path.length() && path.charAt(startIndex) == '/'; ++startIndex) {
        }
        if (startIndex != path.length()) {
            for (endIndex = path.length(); endIndex > 0 && path.charAt(endIndex - 1) == '/'; --endIndex) {
            }
        }
        if (startIndex == path.length()) {
            return "";
        }
        if (startIndex == 0 && endIndex == path.length()) {
            return path;
        }
        if (endIndex == path.length()) {
            return path.substring(startIndex);
        }
        return path.substring(startIndex, endIndex);
    }

    private static String firstPathPart(String path) {
        String stripped = FileUtil.stripSlashes(path);
        if (path.charAt(0) == '/') {
            return "/" + stripped;
        }
        return stripped;
    }
}

