/*
 * Decompiled with CFR 0.152.
 */
package edu.wisc.library.ocfl.core.storage.filesystem;

import com.fasterxml.jackson.databind.ObjectMapper;
import edu.wisc.library.ocfl.api.exception.CorruptObjectException;
import edu.wisc.library.ocfl.api.exception.InvalidInventoryException;
import edu.wisc.library.ocfl.api.exception.OcflIOException;
import edu.wisc.library.ocfl.api.exception.RepositoryConfigurationException;
import edu.wisc.library.ocfl.api.model.OcflVersion;
import edu.wisc.library.ocfl.api.util.Enforce;
import edu.wisc.library.ocfl.core.ObjectPaths;
import edu.wisc.library.ocfl.core.extension.OcflExtensionConfig;
import edu.wisc.library.ocfl.core.extension.OcflExtensionRegistry;
import edu.wisc.library.ocfl.core.extension.storage.layout.OcflLayout;
import edu.wisc.library.ocfl.core.extension.storage.layout.OcflStorageLayoutExtension;
import edu.wisc.library.ocfl.core.util.FileUtil;
import edu.wisc.library.ocfl.core.util.NamasteTypeFile;
import edu.wisc.library.ocfl.core.util.UncheckedFiles;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.CopyOption;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.Map;
import java.util.concurrent.atomic.AtomicReference;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileSystemOcflStorageInitializer {
    private static final Logger LOG = LoggerFactory.getLogger(FileSystemOcflStorageInitializer.class);
    private static final String OBJECT_MARKER_PREFIX = "0=ocfl_object";
    private final ObjectMapper objectMapper;

    public FileSystemOcflStorageInitializer(ObjectMapper objectMapper) {
        this.objectMapper = Enforce.notNull(objectMapper, "objectMapper cannot be null");
    }

    public OcflStorageLayoutExtension initializeStorage(Path repositoryRoot, OcflVersion ocflVersion, OcflExtensionConfig layoutConfig) {
        Enforce.notNull(repositoryRoot, "repositoryRoot cannot be null");
        Enforce.notNull(ocflVersion, "ocflVersion cannot be null");
        if (!Files.exists(repositoryRoot, new LinkOption[0])) {
            UncheckedFiles.createDirectories(repositoryRoot);
        } else {
            Enforce.expressionTrue(Files.isDirectory(repositoryRoot, new LinkOption[0]), repositoryRoot, "repositoryRoot must be a directory");
        }
        OcflStorageLayoutExtension layoutExtension = !FileUtil.hasChildren(repositoryRoot) ? this.initNewRepo(repositoryRoot, ocflVersion, layoutConfig) : this.validateExistingRepo(repositoryRoot, ocflVersion, layoutConfig);
        LOG.info("OCFL repository is configured to use OCFL storage layout extension {} implemented by {}", (Object)layoutExtension.getExtensionName(), (Object)layoutExtension.getClass());
        return layoutExtension;
    }

    private OcflStorageLayoutExtension validateExistingRepo(Path repositoryRoot, OcflVersion ocflVersion, OcflExtensionConfig layoutConfig) {
        this.validateOcflVersion(repositoryRoot, ocflVersion);
        OcflLayout ocflLayout = this.readOcflLayout(repositoryRoot);
        if (ocflLayout == null) {
            LOG.debug("OCFL layout extension not specified");
            return this.validateLayoutByInspection(repositoryRoot, layoutConfig);
        }
        LOG.debug("OCFL layout extension: {}", (Object)ocflLayout.getExtension());
        return this.validateLayoutByConfig(repositoryRoot, ocflLayout, layoutConfig);
    }

    private void validateOcflVersion(Path repositoryRoot, OcflVersion ocflVersion) {
        OcflVersion existingOcflVersion = null;
        for (File file : repositoryRoot.toFile().listFiles()) {
            if (!file.isFile() || !file.getName().startsWith("0=")) continue;
            existingOcflVersion = OcflVersion.fromOcflVersionString(file.getName().substring(2));
            break;
        }
        if (existingOcflVersion == null) {
            throw new RepositoryConfigurationException("OCFL root is missing its namaste file, eg. 0=ocfl_1.0.");
        }
        if (existingOcflVersion != ocflVersion) {
            throw new RepositoryConfigurationException(String.format("OCFL version mismatch. Expected: %s; Found: %s", new Object[]{ocflVersion, existingOcflVersion}));
        }
    }

    private OcflStorageLayoutExtension validateLayoutByConfig(Path repositoryRoot, OcflLayout ocflLayout, OcflExtensionConfig layoutConfig) {
        OcflStorageLayoutExtension layoutExtension = this.loadLayoutExtension(ocflLayout.getExtension());
        OcflExtensionConfig expectedConfig = this.readLayoutConfig(repositoryRoot, ocflLayout, layoutExtension.getExtensionConfigClass());
        if (layoutConfig != null && !layoutConfig.equals(expectedConfig)) {
            throw new RepositoryConfigurationException(String.format("Storage layout configuration does not match. On disk: %s; Configured: %s", expectedConfig, layoutConfig));
        }
        layoutExtension.init(expectedConfig);
        return layoutExtension;
    }

    private OcflStorageLayoutExtension validateLayoutByInspection(Path repositoryRoot, OcflExtensionConfig layoutConfig) {
        Path actualPath;
        String objectId;
        Path expectedPath;
        if (layoutConfig == null) {
            throw new RepositoryConfigurationException(String.format("No storage layout configuration is defined in the OCFL repository at %s. Layout must be configured programmatically.", repositoryRoot));
        }
        OcflStorageLayoutExtension layoutExtension = this.loadAndInitLayoutExtension(layoutConfig);
        Path objectRoot = this.identifyRandomObjectRoot(repositoryRoot);
        if (objectRoot != null && !(expectedPath = Paths.get(layoutExtension.mapObjectId(objectId = this.extractObjectId(ObjectPaths.inventoryPath(objectRoot))), new String[0])).equals(actualPath = repositoryRoot.relativize(objectRoot))) {
            throw new RepositoryConfigurationException(String.format("The OCFL client was configured to use the following layout: %s. This layout does not match the layout of existing objects in the repository. Found object %s stored at %s, but was expecting it to be stored at %s.", layoutConfig, objectId, actualPath, expectedPath));
        }
        return layoutExtension;
    }

    private Path identifyRandomObjectRoot(Path root) {
        final AtomicReference ref = new AtomicReference();
        try {
            Files.walkFileTree(root, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

                @Override
                public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                    if (file.getFileName().toString().startsWith(FileSystemOcflStorageInitializer.OBJECT_MARKER_PREFIX)) {
                        ref.set(file.getParent());
                        return FileVisitResult.TERMINATE;
                    }
                    return super.visitFile(file, attrs);
                }
            });
        }
        catch (IOException e2) {
            throw new OcflIOException(e2);
        }
        return (Path)ref.get();
    }

    private String extractObjectId(Path inventoryPath) {
        if (Files.notExists(inventoryPath, new LinkOption[0])) {
            throw new CorruptObjectException("Missing inventory at " + inventoryPath);
        }
        Map map = this.read(inventoryPath, Map.class);
        Object id = map.get("id");
        if (id == null) {
            throw new InvalidInventoryException(String.format("Inventory file at %s does not contain an id.", inventoryPath));
        }
        return (String)id;
    }

    private OcflStorageLayoutExtension initNewRepo(Path repositoryRoot, OcflVersion ocflVersion, OcflExtensionConfig layoutConfig) {
        Enforce.notNull(layoutConfig, "layoutConfig cannot be null when initializing a new repo");
        LOG.info("Initializing new OCFL repository at {}", (Object)repositoryRoot);
        OcflStorageLayoutExtension layoutExtension = this.loadAndInitLayoutExtension(layoutConfig);
        try {
            new NamasteTypeFile(ocflVersion.getOcflVersion()).writeFile(repositoryRoot);
            this.writeOcflSpec(repositoryRoot, ocflVersion);
            this.writeOcflLayout(repositoryRoot, layoutConfig, layoutExtension.getDescription());
            return layoutExtension;
        }
        catch (RuntimeException e2) {
            LOG.error("Failed to initialize OCFL repository at {}", (Object)repositoryRoot, (Object)e2);
            FileUtil.safeDeleteDirectory(repositoryRoot);
            throw e2;
        }
    }

    private void writeOcflSpec(Path repositoryRoot, OcflVersion ocflVersion) {
        String ocflSpecFile = ocflVersion.getOcflVersion() + ".txt";
        try (InputStream ocflSpecStream = this.getClass().getClassLoader().getResourceAsStream(ocflSpecFile);){
            Files.copy(ocflSpecStream, repositoryRoot.resolve(ocflSpecFile), new CopyOption[0]);
        }
        catch (IOException e2) {
            throw new OcflIOException(e2);
        }
    }

    private void writeOcflLayout(Path repositoryRoot, OcflExtensionConfig layoutConfig, String description) {
        OcflLayout spec = new OcflLayout().setExtension(layoutConfig.getExtensionName()).setDescription(description);
        try {
            this.objectMapper.writeValue(this.ocflLayoutPath(repositoryRoot).toFile(), (Object)spec);
            if (layoutConfig.hasParameters()) {
                this.objectMapper.writeValue(this.layoutExtensionConfigPath(repositoryRoot, layoutConfig.getExtensionName()).toFile(), (Object)layoutConfig);
            }
        }
        catch (IOException e2) {
            throw new OcflIOException(e2);
        }
    }

    private OcflStorageLayoutExtension loadAndInitLayoutExtension(OcflExtensionConfig layoutConfig) {
        OcflStorageLayoutExtension layoutExtension = this.loadLayoutExtension(layoutConfig.getExtensionName());
        layoutExtension.init(layoutConfig);
        return layoutExtension;
    }

    private OcflStorageLayoutExtension loadLayoutExtension(String extensionName) {
        return (OcflStorageLayoutExtension)OcflExtensionRegistry.lookup(extensionName).orElseThrow(() -> new IllegalStateException(String.format("Failed to find an implementation for storage layout extension %s", extensionName)));
    }

    private OcflLayout readOcflLayout(Path repositoryRoot) {
        Path layoutPath = this.ocflLayoutPath(repositoryRoot);
        if (Files.exists(layoutPath, new LinkOption[0])) {
            return this.read(layoutPath, OcflLayout.class);
        }
        return null;
    }

    private OcflExtensionConfig readLayoutConfig(Path repositoryRoot, OcflLayout ocflLayout, Class<? extends OcflExtensionConfig> clazz) {
        Path configPath = this.layoutExtensionConfigPath(repositoryRoot, ocflLayout.getExtension());
        if (Files.exists(configPath, new LinkOption[0])) {
            return this.read(configPath, clazz);
        }
        return this.initClass(clazz);
    }

    private <T> T initClass(Class<T> clazz) {
        try {
            return clazz.getDeclaredConstructor(new Class[0]).newInstance(new Object[0]);
        }
        catch (Exception e2) {
            throw new RepositoryConfigurationException(String.format("Failed to init OCFL storage layout extension configuration class %s", clazz), e2);
        }
    }

    private <T> T read(Path path, Class<T> clazz) {
        try {
            return this.objectMapper.readValue(path.toFile(), clazz);
        }
        catch (IOException e2) {
            throw new OcflIOException(e2);
        }
    }

    private Path ocflLayoutPath(Path repositoryRoot) {
        return repositoryRoot.resolve("ocfl_layout.json");
    }

    private Path layoutExtensionConfigPath(Path repositoryRoot, String extensionName) {
        return repositoryRoot.resolve(extensionName + ".json");
    }
}

