/*
 * Decompiled with CFR 0.152.
 */
package edu.wisc.library.ocfl.core.validation;

import edu.wisc.library.ocfl.api.OcflConstants;
import edu.wisc.library.ocfl.api.exception.CorruptObjectException;
import edu.wisc.library.ocfl.api.exception.InvalidInventoryException;
import edu.wisc.library.ocfl.api.model.DigestAlgorithm;
import edu.wisc.library.ocfl.api.model.VersionNum;
import edu.wisc.library.ocfl.api.util.Enforce;
import edu.wisc.library.ocfl.core.model.Inventory;
import edu.wisc.library.ocfl.core.model.User;
import edu.wisc.library.ocfl.core.model.Version;
import java.util.Collection;
import java.util.HashSet;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

public final class InventoryValidator {
    private static final VersionNum VERSION_ZERO = VersionNum.fromString("v0");
    private static final Pattern CONTENT_DIR_PATTERN = Pattern.compile(".*[/\\\\].*");

    private InventoryValidator() {
    }

    public static Inventory validateShallow(Inventory inventory) {
        Enforce.notNull(inventory, "inventory cannot be null");
        InventoryValidator.notBlank(inventory.getId(), () -> "Object ID cannot be blank");
        InventoryValidator.notNull(inventory.getHead(), InventoryValidator.prefix(inventory, () -> "HEAD cannot be null"));
        InventoryValidator.isTrue(!inventory.getHead().equals(VERSION_ZERO), InventoryValidator.prefix(inventory, () -> "HEAD version must be greater than v0"));
        InventoryValidator.notNull((Object)inventory.getType(), InventoryValidator.prefix(inventory, () -> "Type cannot be null"));
        InventoryValidator.validateDigestAlgorithm(inventory);
        InventoryValidator.validateContentDirectory(inventory);
        InventoryValidator.notNull(inventory.getManifest(), InventoryValidator.prefix(inventory, () -> "Manifest cannot be null"));
        InventoryValidator.validateVersions(inventory, false);
        return inventory;
    }

    public static Inventory validateDeep(Inventory inventory) {
        Enforce.notNull(inventory, "inventory cannot be null");
        InventoryValidator.notBlank(inventory.getId(), () -> "Object ID cannot be blank");
        InventoryValidator.notNull(inventory.getHead(), InventoryValidator.prefix(inventory, () -> "HEAD cannot be null"));
        InventoryValidator.isTrue(!inventory.getHead().equals(VERSION_ZERO), InventoryValidator.prefix(inventory, () -> "HEAD version must be greater than v0"));
        InventoryValidator.notNull((Object)inventory.getType(), InventoryValidator.prefix(inventory, () -> "Type cannot be null"));
        InventoryValidator.validateDigestAlgorithm(inventory);
        InventoryValidator.validateContentDirectory(inventory);
        InventoryValidator.validateVersionNumbers(inventory);
        InventoryValidator.validateFixity(inventory);
        InventoryValidator.notNull(inventory.getManifest(), InventoryValidator.prefix(inventory, () -> "Manifest cannot be null"));
        InventoryValidator.validateVersions(inventory, true);
        InventoryValidator.validateManifest(inventory);
        return inventory;
    }

    public static void validateVersionStates(Inventory currentInventory, Inventory previousInventory) {
        VersionNum current = previousInventory.getHead();
        Map currentVersions = currentInventory.getVersions().keySet().stream().collect(Collectors.toMap(Function.identity(), Function.identity()));
        Map previousVersions = previousInventory.getVersions().keySet().stream().collect(Collectors.toMap(Function.identity(), Function.identity()));
        while (true) {
            InventoryValidator.validateVersionNumber(currentInventory, (VersionNum)currentVersions.get(current), (VersionNum)previousVersions.get(current));
            Map<String, Set<String>> currentState = currentInventory.getVersion(current).getState();
            Map<String, Set<String>> previousState = previousInventory.getVersion(current).getState();
            if (!Objects.equals(currentState, previousState)) {
                throw new InvalidInventoryException(String.format("In object %s the inventories in version %s and %s define a different state for version %s.", currentInventory.getId(), currentInventory.getHead(), previousInventory.getHead(), current));
            }
            if (VersionNum.V1.equals(current)) break;
            current = current.previousVersionNum();
        }
    }

    public static void validateCompatibleInventories(Inventory currentInventory, Inventory previousInventory) {
        InventoryValidator.areEqual(currentInventory.getId(), previousInventory.getId(), () -> String.format("Object IDs are not the same. Existing: %s; New: %s", previousInventory.getId(), currentInventory.getId()));
        InventoryValidator.areEqual(currentInventory.getType(), previousInventory.getType(), () -> String.format("Inventory types are not the same. Existing: %s; New: %s", previousInventory.getType().getId(), currentInventory.getType().getId()));
        InventoryValidator.areEqual(currentInventory.getDigestAlgorithm(), previousInventory.getDigestAlgorithm(), () -> String.format("Inventory digest algorithms are not the same. Existing: %s; New: %s", previousInventory.getDigestAlgorithm().getOcflName(), currentInventory.getDigestAlgorithm().getOcflName()));
        InventoryValidator.areEqual(currentInventory.getContentDirectory(), previousInventory.getContentDirectory(), () -> String.format("Inventory content directories are not the same. Existing: %s; New: %s", previousInventory.getContentDirectory(), currentInventory.getContentDirectory()));
        if (!Objects.equals(currentInventory.getHead(), previousInventory.nextVersionNum())) {
            throw new InvalidInventoryException(String.format("The new HEAD inventory version must be the next sequential version number. Existing: %s; New: %s", previousInventory.getHead(), currentInventory.getHead()));
        }
        InventoryValidator.validateVersionStates(currentInventory, previousInventory);
    }

    private static void validateFixity(Inventory inventory) {
        Map<DigestAlgorithm, Map<String, Set<String>>> fixityMap = inventory.getFixity();
        if (fixityMap != null) {
            fixityMap.forEach((algorithm, map) -> map.forEach((digest, contentPaths) -> {
                InventoryValidator.notEmpty(contentPaths, InventoryValidator.prefix(inventory, () -> "Fixity content paths cannot be empty"));
                contentPaths.forEach(contentPath -> InventoryValidator.notNull(inventory.getFileId((String)contentPath), InventoryValidator.prefix(inventory, () -> String.format("Fixity entry %s => {%s => %s} does not have a corresponding entry in the manifest block.", algorithm.getOcflName(), digest, contentPath))));
            }));
        }
    }

    private static void validateVersions(Inventory inventory, boolean allVersions) {
        Map<VersionNum, Version> versionMap = inventory.getVersions();
        InventoryValidator.notEmpty(versionMap, InventoryValidator.prefix(inventory, () -> "Versions cannot be empty"));
        for (int i = 1; i <= versionMap.size(); ++i) {
            VersionNum versionNum2 = new VersionNum(i);
            InventoryValidator.notNull(versionMap.get(versionNum2), InventoryValidator.prefix(inventory, () -> String.format("Version %s is missing", versionNum2)));
        }
        VersionNum expectedHead = new VersionNum(versionMap.size());
        InventoryValidator.isTrue(inventory.getHead().equals(expectedHead), InventoryValidator.prefix(inventory, () -> String.format("HEAD must be the latest version. Expected: %s; Was: %s", expectedHead, inventory.getHead())));
        if (!allVersions) {
            InventoryValidator.validateVersion(inventory, versionMap.get(expectedHead), expectedHead);
        } else {
            versionMap.forEach((versionNum, version2) -> InventoryValidator.validateVersion(inventory, version2, versionNum));
        }
    }

    private static void validateVersion(Inventory inventory, Version version2, VersionNum versionNum) {
        HashSet check;
        HashSet iter;
        InventoryValidator.notNull(version2, InventoryValidator.prefix(inventory, () -> String.format("Version %s is missing", versionNum)));
        InventoryValidator.notNull(version2.getCreated(), InventoryValidator.prefix(inventory, () -> String.format("Version created timestamp in version %s cannot be null", versionNum)));
        InventoryValidator.validateUser(inventory, version2.getUser(), versionNum);
        Map<String, Set<String>> state = version2.getState();
        InventoryValidator.notNull(state, InventoryValidator.prefix(inventory, () -> String.format("Version state of version %s cannot be null", versionNum)));
        HashSet directories = new HashSet();
        HashSet files = new HashSet();
        state.forEach((digest, logicalPaths) -> {
            InventoryValidator.notEmpty(logicalPaths, InventoryValidator.prefix(inventory, () -> String.format("Version state logical paths in version %s cannot be empty", versionNum)));
            InventoryValidator.notNull(inventory.getContentPath((String)digest), InventoryValidator.prefix(inventory, () -> String.format("Version state entry %s => %s in version %s does not have a corresponding entry in the manifest block.", digest, logicalPaths, versionNum)));
            logicalPaths.forEach(path -> {
                files.add(path);
                String[] parts = path.split("/");
                for (int i = 0; i < parts.length - 1; ++i) {
                    StringBuilder builder = new StringBuilder(parts[0]);
                    for (int j = 1; j <= i; ++j) {
                        builder.append("/").append(parts[j]);
                    }
                    directories.add(builder.toString());
                }
            });
        });
        if (files.size() > directories.size()) {
            iter = directories;
            check = files;
        } else {
            iter = files;
            check = directories;
        }
        iter.forEach(path -> {
            if (check.contains(path)) {
                throw new InvalidInventoryException(InventoryValidator.prefix(inventory, () -> String.format("In version %s the logical path %s conflicts with another logical path.", versionNum, path)).get());
            }
        });
    }

    private static void validateManifest(Inventory inventory) {
        Set stateFileIds = inventory.getVersions().values().stream().flatMap(v -> v.getState().keySet().stream()).collect(Collectors.toSet());
        inventory.getManifest().keySet().forEach(fileId -> {
            if (!stateFileIds.contains(fileId)) {
                throw new CorruptObjectException(InventoryValidator.prefix(inventory, () -> String.format("Manifest contains an entry for %s, but it is not referenced in any version's state.", fileId)).get());
            }
        });
    }

    private static void validateDigestAlgorithm(Inventory inventory) {
        DigestAlgorithm digestAlgorithm = inventory.getDigestAlgorithm();
        InventoryValidator.notNull(digestAlgorithm, InventoryValidator.prefix(inventory, () -> "Digest algorithm cannot be null"));
        InventoryValidator.isTrue(OcflConstants.ALLOWED_DIGEST_ALGORITHMS.contains(digestAlgorithm), InventoryValidator.prefix(inventory, () -> String.format("Digest algorithm must be one of: %s; Found: %s", OcflConstants.ALLOWED_DIGEST_ALGORITHMS, digestAlgorithm)));
    }

    private static void validateContentDirectory(Inventory inventory) {
        String contentDirectory = inventory.getContentDirectory();
        if (contentDirectory != null) {
            InventoryValidator.notBlank(contentDirectory, InventoryValidator.prefix(inventory, () -> "Content directory cannot be blank"));
            InventoryValidator.isTrue(!CONTENT_DIR_PATTERN.matcher(contentDirectory).matches(), InventoryValidator.prefix(inventory, () -> "Content directory cannot contain / or \\. Found: " + contentDirectory));
        }
    }

    private static void validateUser(Inventory inventory, User user, VersionNum versionNum) {
        if (user != null) {
            InventoryValidator.notBlank(user.getName(), InventoryValidator.prefix(inventory, () -> String.format("User name in version %s cannot be blank", versionNum)));
        }
    }

    private static void validateVersionNumbers(Inventory inventory) {
        int expectedPadding = inventory.getHead().getZeroPaddingWidth();
        inventory.getVersions().keySet().forEach(versionNumber -> {
            if (versionNumber.getZeroPaddingWidth() != expectedPadding) {
                throw new InvalidInventoryException(InventoryValidator.prefix(inventory, () -> String.format("%s is not zero-padded correctly. Expected: %s; Actual: %s", versionNumber, expectedPadding, versionNumber.getZeroPaddingWidth())).get());
            }
        });
    }

    private static void validateVersionNumber(Inventory inventory, VersionNum currentVersion, VersionNum previousVersion) {
        if (!Objects.equals(currentVersion.toString(), previousVersion.toString())) {
            throw new InvalidInventoryException(InventoryValidator.prefix(inventory, () -> String.format("Version number formatting differs: %s vs %s", previousVersion, currentVersion)).get());
        }
    }

    private static Supplier<String> prefix(Inventory inventory, Supplier<String> message) {
        return () -> String.format("Inventory %s %s: %s", inventory.getId(), inventory.getHead(), message.get());
    }

    private static void notNull(Object object, Supplier<String> messageSupplier) {
        if (object == null) {
            throw new InvalidInventoryException(messageSupplier.get());
        }
    }

    private static void notBlank(String value, Supplier<String> messageSupplier) {
        if (value == null || value.isBlank()) {
            throw new InvalidInventoryException(messageSupplier.get());
        }
    }

    private static void notEmpty(Collection<?> collection, Supplier<String> messageSupplier) {
        if (collection == null || collection.isEmpty()) {
            throw new InvalidInventoryException(messageSupplier.get());
        }
    }

    private static void notEmpty(Map<?, ?> map, Supplier<String> messageSupplier) {
        if (map == null || map.isEmpty()) {
            throw new InvalidInventoryException(messageSupplier.get());
        }
    }

    private static void isTrue(boolean test, Supplier<String> messageSupplier) {
        if (!test) {
            throw new InvalidInventoryException(messageSupplier.get());
        }
    }

    private static <T> void areEqual(T left, T right, Supplier<String> messageSupplier) {
        if (!Objects.equals(left, right)) {
            throw new InvalidInventoryException(messageSupplier.get());
        }
    }
}

