/*
 * Decompiled with CFR 0.152.
 */
package edu.wisc.library.ocfl.core.storage.filesystem;

import edu.wisc.library.ocfl.api.OcflFileRetriever;
import edu.wisc.library.ocfl.api.exception.OcflFileAlreadyExistsException;
import edu.wisc.library.ocfl.api.exception.OcflIOException;
import edu.wisc.library.ocfl.api.exception.OcflNoSuchFileException;
import edu.wisc.library.ocfl.api.model.DigestAlgorithm;
import edu.wisc.library.ocfl.api.util.Enforce;
import edu.wisc.library.ocfl.core.storage.common.Listing;
import edu.wisc.library.ocfl.core.storage.common.OcflObjectRootDirIterator;
import edu.wisc.library.ocfl.core.storage.common.Storage;
import edu.wisc.library.ocfl.core.storage.filesystem.FileSystemOcflFileRetriever;
import edu.wisc.library.ocfl.core.storage.filesystem.FileSystemOcflObjectRootDirIterator;
import edu.wisc.library.ocfl.core.util.FileUtil;
import java.io.BufferedInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileSystemStorage
implements Storage {
    private static final Logger LOG = LoggerFactory.getLogger(FileSystemStorage.class);
    private final Path storageRoot;

    public FileSystemStorage(Path storageRoot) {
        this.storageRoot = Enforce.notNull(storageRoot, "storageRoot cannot be null");
    }

    @Override
    public List<Listing> listDirectory(String directoryPath) {
        List<Listing> list;
        block8: {
            Path fullPath = this.storageRoot.resolve(directoryPath);
            Stream<Path> children = Files.list(fullPath);
            try {
                list = children.map(child -> {
                    String name = child.getFileName().toString();
                    if (Files.isRegularFile(child, new LinkOption[0])) {
                        return Listing.file(name);
                    }
                    if (Files.isDirectory(child, new LinkOption[0])) {
                        return Listing.directory(name);
                    }
                    return Listing.other(name);
                }).collect(Collectors.toList());
                if (children == null) break block8;
            }
            catch (Throwable throwable) {
                try {
                    if (children != null) {
                        try {
                            children.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException e2) {
                    throw OcflIOException.from(e2);
                }
            }
            children.close();
        }
        return list;
    }

    @Override
    public List<Listing> listRecursive(String directoryPath) {
        final Path fullPath = this.storageRoot.resolve(directoryPath);
        final ArrayList<Listing> listings = new ArrayList<Listing>();
        try {
            Files.walkFileTree(fullPath, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

                @Override
                public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                    if (attrs.isRegularFile()) {
                        listings.add(this.createListing(Listing.Type.File, file));
                    } else {
                        listings.add(this.createListing(Listing.Type.Other, file));
                    }
                    return super.visitFile(file, attrs);
                }

                @Override
                public FileVisitResult postVisitDirectory(Path dir, IOException exc) throws IOException {
                    if (FileUtil.isDirEmpty(dir) && !dir.equals(fullPath)) {
                        listings.add(this.createListing(Listing.Type.Directory, dir));
                    }
                    return super.postVisitDirectory(dir, exc);
                }

                private Listing createListing(Listing.Type type, Path file) {
                    String relative = FileUtil.pathToStringStandardSeparator(fullPath.relativize(file));
                    return new Listing(type, relative);
                }
            });
        }
        catch (IOException e2) {
            throw OcflIOException.from(e2);
        }
        return listings;
    }

    @Override
    public boolean directoryIsEmpty(String directoryPath) {
        boolean bl;
        block8: {
            Path fullPath = this.storageRoot.resolve(directoryPath);
            Stream<Path> children = Files.list(fullPath);
            try {
                bl = children.findAny().isEmpty();
                if (children == null) break block8;
            }
            catch (Throwable throwable) {
                try {
                    if (children != null) {
                        try {
                            children.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException e2) {
                    throw OcflIOException.from(e2);
                }
            }
            children.close();
        }
        return bl;
    }

    @Override
    public OcflObjectRootDirIterator iterateObjects() {
        return new FileSystemOcflObjectRootDirIterator(this.storageRoot);
    }

    @Override
    public boolean fileExists(String filePath) {
        return Files.exists(this.storageRoot.resolve(filePath), new LinkOption[0]);
    }

    @Override
    public InputStream read(String filePath) {
        try {
            return new BufferedInputStream(Files.newInputStream(this.storageRoot.resolve(filePath), new OpenOption[0]));
        }
        catch (IOException e2) {
            throw OcflIOException.from(e2);
        }
    }

    @Override
    public String readToString(String filePath) {
        String string;
        block8: {
            InputStream stream = this.read(filePath);
            try {
                string = new String(stream.readAllBytes());
                if (stream == null) break block8;
            }
            catch (Throwable throwable) {
                try {
                    if (stream != null) {
                        try {
                            stream.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException e2) {
                    throw OcflIOException.from(e2);
                }
            }
            stream.close();
        }
        return string;
    }

    @Override
    public OcflFileRetriever readLazy(String filePath, DigestAlgorithm algorithm, String digest) {
        Path fullPath = this.storageRoot.resolve(filePath);
        return new FileSystemOcflFileRetriever(fullPath, algorithm, digest);
    }

    @Override
    public void write(String filePath, byte[] content, String mediaType) {
        Path fullPath = this.storageRoot.resolve(filePath);
        try {
            Files.write(fullPath, content, StandardOpenOption.WRITE, StandardOpenOption.CREATE_NEW);
        }
        catch (IOException e2) {
            throw OcflIOException.from(e2);
        }
    }

    @Override
    public void createDirectories(String path) {
        Path fullPath = this.storageRoot.resolve(path);
        try {
            Files.createDirectories(fullPath, new FileAttribute[0]);
        }
        catch (IOException e2) {
            throw OcflIOException.from(e2);
        }
    }

    @Override
    public void copyDirectoryOutOf(String source, Path destination) {
        Path fullPath = this.storageRoot.resolve(source);
        FileUtil.recursiveCopy(fullPath, destination, new StandardCopyOption[0]);
    }

    @Override
    public void copyFileInto(Path source, String destination, String mediaType) {
        Path dstPath = this.storageRoot.resolve(destination);
        try {
            Files.copy(source, dstPath, StandardCopyOption.REPLACE_EXISTING);
        }
        catch (IOException e2) {
            throw OcflIOException.from(e2);
        }
    }

    @Override
    public void copyFileInternal(String sourceFile, String destinationFile) {
        Path srcPath = this.storageRoot.resolve(sourceFile);
        Path dstPath = this.storageRoot.resolve(destinationFile);
        try {
            Files.copy(srcPath, dstPath, StandardCopyOption.REPLACE_EXISTING);
        }
        catch (IOException e2) {
            throw OcflIOException.from(e2);
        }
    }

    @Override
    public void moveDirectoryInto(Path source, String destination) {
        Path dstPath = this.storageRoot.resolve(destination);
        try {
            FileUtil.moveDirectory(source, dstPath);
        }
        catch (FileAlreadyExistsException e2) {
            throw new OcflFileAlreadyExistsException(e2);
        }
    }

    @Override
    public void moveDirectoryInternal(String source, String destination) {
        Path srcPath = this.storageRoot.resolve(source);
        Path dstPath = this.storageRoot.resolve(destination);
        try {
            FileUtil.moveDirectory(srcPath, dstPath);
        }
        catch (IllegalArgumentException e2) {
            if (e2.getMessage() != null && e2.getMessage().contains("Source must exist")) {
                throw new OcflNoSuchFileException(String.format("Directory %s does not exist", source));
            }
        }
        catch (FileAlreadyExistsException e3) {
            throw new OcflFileAlreadyExistsException(e3);
        }
    }

    @Override
    public void deleteDirectory(String path) {
        Path fullPath = this.storageRoot.resolve(path);
        FileUtil.deleteDirectory(fullPath);
    }

    @Override
    public void deleteFile(String path) {
        Path fullPath = this.storageRoot.resolve(path);
        try {
            Files.deleteIfExists(fullPath);
        }
        catch (IOException e2) {
            throw OcflIOException.from(e2);
        }
    }

    @Override
    public void deleteFiles(Collection<String> paths) {
        ArrayList<String> failed = new ArrayList<String>();
        for (String path : paths) {
            try {
                this.deleteFile(path);
            }
            catch (RuntimeException e2) {
                LOG.warn("Failed to delete file: {}", (Object)path, (Object)e2);
                failed.add(path);
            }
        }
        if (!failed.isEmpty()) {
            throw new OcflIOException(String.format("Failed to delete files: %s", failed));
        }
    }

    @Override
    public void deleteEmptyDirsDown(String path) {
        Path fullPath = this.storageRoot.resolve(path);
        FileUtil.deleteEmptyDirs(fullPath);
    }

    @Override
    public void deleteEmptyDirsUp(String path) {
        Path fullPath = this.storageRoot.resolve(path);
        FileUtil.deleteDirAndParentsIfEmpty(fullPath);
    }
}

