package org.ferris.bible.servlet;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.GregorianCalendar;
import java.util.List;
import java.util.TreeSet;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.ferris.bible.store.Store;
import org.ferris.bible.store.StoreFactory;
import org.ferris.bible.store.StoreItem;

import com.simplerss.dataobject.Channel;
import com.simplerss.dataobject.Guid;
import com.simplerss.dataobject.Item;
import com.simplerss.dataobject.Source;
import com.simplerss.helper.RSSBuilder;

/**
 * http://sourceforge.net/projects/simple-rss/
 */
public class DailyBibleReadingRssServlet extends HttpServlet 
{

	private static final long serialVersionUID = -6911931194683267534L;
	
	private static final String MyEmail = "mjremijan@yahoo.com (Mike Remijan)";
	private static final String ManagingEditor = MyEmail;
	private static final String WebMaster = MyEmail;
	private static final String Author = MyEmail;
	private static final String Category = "Bible New Living Translation";
	private static final String Title = "Daily Bible Reading";
	private static       URL    Link = null;
	private static       URL    Source = null;
	
	static {
		String s = "http://ferris.dnsalias.net:8888/bible/rss/";
		String t = s + "DailyBibleReading.xml";
		try {
			Link = new URL(s);
			Source = new URL(t);
		} catch (MalformedURLException e) {
			throw new ExceptionInInitializerError(e);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected void doGet(HttpServletRequest req, HttpServletResponse resp)
	throws ServletException, IOException
	{
		
		ByteArrayOutputStream baos = new ByteArrayOutputStream();
		go(baos);
		baos.flush();
		
		resp.setContentType("application/rss+xml");
		resp.setContentLength(baos.size());
		resp.getOutputStream().write(baos.toByteArray());
		baos.close();
		resp.getOutputStream().flush();
	}
	
	public void go(OutputStream os)
	throws IOException
	{
		Store store
			= StoreFactory.getStore();
	
		List<StoreItem> items
			= store.getAll();
		if (items.isEmpty()) { return; }
		
		Calendar now
			= Calendar.getInstance();

		Channel c = new Channel();
		c.setCategory(Category);
		c.setCloud(null);
		c.setCopyright("Copyright, BibleGateway.com");
		c.setDescription("Read the Bible this year");
		c.setDocs(new URL("http://cyber.law.harvard.edu/rss/rss.html"));
		c.setGenerator("Ferris Bible JMCE DailyBibleReadingRssServlet v" + new Version().getVersion());
		c.setImage(null);
		c.setItem(getItems(items));
		c.setLanguage("en-us");
		c.setLastBuildDate(now);
		c.setLink(Link);
		c.setManagingEditor(ManagingEditor);
		c.setPubDate(getLatestReadingDay(items));
		//c.setRating(null)
		//c.setSkipDays(null);
		//c.setSkipHours(null);
		c.setTextInput(null);
		c.setTitle(Title);
		c.setTtl(60);
		c.setWebMaster(WebMaster);

		os.write(RSSBuilder.getRSSDocument(c).getBytes("utf-8"));
	}
	
	
	/** 
	 * Get the Item objects for the RSS feed.
	 */
	private Item[] getItems(List<StoreItem> storeItems)
	{
		Item[] items = new Item[storeItems.size()];
		int itemsIdx = 0;
		
		for(StoreItem storeItem : storeItems)
		{
			Item item = new Item();
			
			item.setAuthor(Author);
			item.setCategory(Category);
			//item.setComments(null);
			item.setDescription(storeItem.getReading());
			//item.setEnclosure(null);
			item.setGuid(getGuid(storeItem));
			//item.setLink(null);
			item.setPubDate(getCalendar(storeItem));
			item.setSource(getSource(storeItem));
			item.setTitle(getTitle(storeItem));
			
			items[itemsIdx] = item;
			itemsIdx++;
		}
		
		return items;
	}
	
	/**
	 * @param dailyReadings
	 * @return
	 */
	private Guid getGuid(StoreItem storeItem)
	{
		Guid guid = new Guid();
		guid.setPermaLink(false);
		try {
			guid.setUrl(new URL(Source.toString() + "#" + yyyyMMdd(storeItem)));
		} catch (MalformedURLException e) {
			throw new RuntimeException(e.getMessage(),e);
		}
		return guid;
	}

	/**
	 * Get a Source for the DailyReadings
	 */
	private Source getSource(StoreItem storeItem)
	{
		Source s = new Source();		
		s.setTitle(getTitle(storeItem));
		s.setUrl(Source);
		return s;
	}
	
	/**
	 * Get a title for the DailyReadings
	 */
	private String getTitle(StoreItem storeItem)
	{
		return 
			Title 
			+ " for " 
			//+ new SimpleDateFormat("MM/dd/yyyy").format(getCalendar(storeItem).getTime());		
			+ new SimpleDateFormat("MM/dd").format(getCalendar(storeItem).getTime());
	}
	
	
	/**
	 * Get the latest reading day.
	 */
	private Calendar getLatestReadingDay(List<StoreItem> items)
	{
		// Create a sorted set.
		TreeSet<String> set = new TreeSet<String>();
		
		// Add to the set the dates of the daily readings
		for(StoreItem item : items)
		{
			set.add(
				yyyyMMdd(item)
			);
		}
		
		// The last item in the sorted set is the greatest
		// value, i.e. the latest date.  Convert this into
		// a valid Calendar object and then return it.
		return getCalendar(set.last());
	}
	
	/**
	 * Get DailyReadings as yyyyMMdd string.
	 */
	private String yyyyMMdd(StoreItem storeItem) {
		return
			  storeItem.getYear()
			+ ((storeItem.getMonth() < 10) ? "0" : "") + storeItem.getMonth()
			+ ((storeItem.getDay()   < 10) ? "0" : "") + storeItem.getDay()
		;
	}
	
	
	/**
	 * Get yyyyMMdd string as Calendar
	 * 
	 * @param yyyyMMdd
	 * @return
	 */
	private Calendar getCalendar(String yyyyMMdd)
	{
		GregorianCalendar gc = new GregorianCalendar();
		try {
			gc.setTime(new SimpleDateFormat("yyyyMMdd").parse(yyyyMMdd));
		} catch (Throwable t) {
			throw new RuntimeException(t.getMessage(), t);
		}
		gc.clear(Calendar.HOUR);
		gc.clear(Calendar.MINUTE);
		gc.clear(Calendar.SECOND);
		gc.clear(Calendar.MILLISECOND);
		
		return gc;
	}
	
	/**
	 * Get DailyReadings as Calendar
	 */
	private Calendar getCalendar(StoreItem storeItem) {
		return getCalendar(yyyyMMdd(storeItem));
	}
}
