package org.ferris.journal.ejb.account;

import javax.ejb.Local;
import javax.ejb.Stateless;
import javax.persistence.EntityManager;
import javax.persistence.NoResultException;
import javax.persistence.PersistenceContext;
import javax.persistence.Query;

import org.apache.log4j.Logger;
import org.ferris.constraint.Check;
import org.ferris.constraint.Constraint;
import org.ferris.constraint.GreaterThanOrEqualTo;
import org.ferris.constraint.NotNull;
import org.ferris.journal.ejb.lang.DomainException;
import org.ferris.journal.ejb.lang.DomainExceptionCode;

@Stateless(name="AccountLocal")
@Local(AccountLocal.class)
public class AccountEjb implements AccountLocal
{
	private Logger log = Logger.getLogger(getClass());
	
	
	@PersistenceContext(unitName="JournalPersistenceUnit")
	private EntityManager entityManager;
	
	
	/**
	 * Return the EntityManager
	 */
	private EntityManager getEntityManager() {
		log.info("Enter: getEntityManager()");
		return entityManager;
	}
	
	
	/** 
	 * Default no-arg constructor
	 */
	public AccountEjb() {}
	
	
	/**
	 * Set the EntityManager, useful for testing purposes.
	 */
	public AccountEjb(EntityManager entityManager) {
		this.entityManager = entityManager;
	}
	
	
	/**
	 * Insert account data into db.
	 */
	public Account insert(Account account) 
	{
		log.info("Enter: insert(account) " + account);
	
		log.info("Not null constraint");
		Constraint<Account> constraint
			= new NotNull<Account>(); 
		
		log.info("Constraint check");
		Check check
			= constraint.check(account);
		
		if (check != null) {			
			if (check.equals(Check.IsNull)) {
				DomainException e
					= new DomainException(DomainExceptionCode.DOMAIN00019);
				log.info(e.toString());
				throw e;
			}			
		}
		
		EntityManager em = getEntityManager();
		
		log.info("Persist");
		em.persist(account);
		
		log.info("Return");
		return findId(account.getId());
	}


	public Account findId(long id) 
	{
		log.info("Enter: findId("+id+")");
		
		log.info("Greater than or equal to constraint");		
		Constraint<Long> constraint
			= new GreaterThanOrEqualTo<Long>(1L);
		
		log.info("Constraint check");
		Check check 
			= constraint.check(id);
		if (check != null) {
			if (check.equals(Check.LessThan)) {
				DomainException e
					= new DomainException(DomainExceptionCode.DOMAIN00001, id, 1);
				log.info(e.toString());
				throw e;
			}
		}
		
		EntityManager em
			= getEntityManager();
		
		log.info("Find account");
		Account dbData
			= em.find(Account.class, id);
		
		log.info("Return");
		return dbData;
	}


	public Account findUsername(String username) 
	{
		log.info("Enter: findUsername("+username+")");
		
		log.info("Not null constraint");
		Constraint<String> constraint
			= new NotNull<String>();
		
		log.info("Constraint check");
		Check check 
			= constraint.check(username);
		
		
		if (check != null) {
			DomainException e
				=  new DomainException(DomainExceptionCode.DOMAIN00002);
			log.info(e.toString());
			throw e;
		}		
		
		EntityManager em
			= getEntityManager();
		
		log.info("Create query");
		Query query 
			= em.createQuery("select account from Account account where username = ?1" );
		
		log.info("Set parameter: " + username);
		query.setParameter(1, username);
		
		Account dbData = null;
		try {
			log.info("Run query");
			dbData = (Account)query.getSingleResult();
		} catch (NoResultException e) {}
		
		log.info("Return");
		return dbData;
	}
}
