package org.ferris.journal.ejb.journal;

import java.util.ArrayList;
import java.util.List;

import javax.ejb.Local;
import javax.ejb.Stateless;
import javax.persistence.EntityManager;
import javax.persistence.NoResultException;
import javax.persistence.PersistenceContext;
import javax.persistence.Query;

import org.apache.log4j.Logger;
import org.ferris.constraint.Check;
import org.ferris.constraint.Constraint;
import org.ferris.constraint.GreaterThan;
import org.ferris.constraint.NotNull;
import org.ferris.journal.ejb.lang.DomainException;
import org.ferris.journal.ejb.lang.DomainExceptionCode;

@Stateless(name="JournalLocal")
@Local(JournalLocal.class)
public class JournalEjb implements JournalLocal
{
	private Logger log = Logger.getLogger(getClass());
	
	/**
	 * Return EntityManager
	 * @return
	 */
	private EntityManager getEntityManager() {
		log.info("Enter: getEntityManager()");
		return entityManager;
	}
	
	@PersistenceContext(unitName="JournalPersistenceUnit")
	private EntityManager entityManager;
	
	
	/** 
	 * Default no-arg constructor
	 */
	public JournalEjb() {}
	
	
	/**
	 * Set EntityManager
	 */
	public JournalEjb(EntityManager entityManager) {
		this.entityManager = entityManager;
	}
	
	
	public Journal insert(Journal journal) 
	{
		log.info("Enter: insert(journal) " + journal);
		
		log.info("Not null constraint");
		Constraint<Journal> constraint
			= new NotNull<Journal>(); 
		
		log.info("Contraint check");
		Check check
			= constraint.check(journal);
		
		if (check != null) {			
			if (check.equals(Check.IsNull)) {
				DomainException e = new DomainException(DomainExceptionCode.DOMAIN00018);
				log.info(e.toString());
				throw e;
			}			
		}
				
		EntityManager em = getEntityManager();
		
		log.info("Persist");		
		em.persist(journal);
		
		return findById(journal.getAccount().getId(), journal.getId());
	}
	
	
	public Journal update(Journal journal) throws DomainException 
	{
		log.info("Enter: update(journal) " + journal);
		
		log.info("Not null constraint");
		Constraint<Journal> constraint
			= new NotNull<Journal>(); 
		
		log.info("Constraint check");
		Check check
			= constraint.check(journal);
		
		if (check != null) {
			if (check.equals(Check.IsNull)) {
				DomainException e = new DomainException(DomainExceptionCode.DOMAIN00020);
				log.info(e.toString());
				throw e;
			}			
		}
		
		EntityManager em = getEntityManager();
		
		log.info("Merge");
		em.merge(journal);
		
		journal = findById(journal.getAccount().getId(), journal.getId());
				
		log.info("Constraint check");
		check
			= constraint.check(journal);
		
		if (check != null) {
			if (check.equals(Check.IsNull)) {
				DomainException e = new DomainException(DomainExceptionCode.DOMAIN00021);
				log.info(e.toString());
				throw e;
			}			
		}
		
		return journal;
	}

	
	/**
	 * Find a journal by it's unique identifer.
	 * 
	 * @param accountId - required, can not be null
	 * @param x - required, can not be null.
	 * @return Returns the Journal object found or null
	 * if data is not found.
	 * 
	 * @throws DomainException
	 */
	public Journal findById(long accountId, long id) 
	throws DomainException
	{
		log.info("Enter: findId("+accountId+","+id+")");
			
		log.info("Greater than constraint");
		Long min = 0L;
		Constraint<Long> constraint
			= new GreaterThan<Long>(min); 
		
		// accountId
		{
			long x = accountId;
			
			log.info("Constraint check (accountId)");
			Check check
				= constraint.check(x);
			
			if (check != null) {
				if (check.equals(Check.LessThan)) {
					DomainException e = new DomainException(DomainExceptionCode.DOMAIN00022, x, min);
					log.info(e.toString());
					throw e;
				}			
				else 
				if (check.equals(Check.EqualToMin)) {
					DomainException e = new DomainException(DomainExceptionCode.DOMAIN00022, x, min);
					log.info(e.toString());
					throw e;
				}
			}
		}
		
		// id
		{
			long x = id;
			
			log.info("Constraint check (id)");
			Check check
				= constraint.check(x);
			
			if (check != null) {
				if (check.equals(Check.LessThan)) {
					DomainException e = new DomainException(DomainExceptionCode.DOMAIN00016, x, min);
					log.info(e.toString());
					throw e;
				}			
				else 
				if (check.equals(Check.EqualToMin)) {
					DomainException e = new DomainException(DomainExceptionCode.DOMAIN00016, x, min);
					log.info(e.toString());
					throw e;
				}
			}
		}
		
		log.info("Create query");
		Query query 
			= getEntityManager().createQuery(" select journal from Journal journal where account.id=?1 and id=?2 ");
		
		// account id
		log.info("Set parameter: " + accountId);
		query.setParameter(1, accountId);
		
		// id
		log.info("Set parameter: " + id);
		query.setParameter(2, id);
		
		try {
			log.info("Run query");
			return (Journal)query.getSingleResult();
		} catch (NoResultException e) {
			return null;
		}
	}


	/**
	 * Find a list of journals by account and the journal's active status.
	 * 
	 * @param accountId - Required, can not be null.
	 * 
	 * @param isActive - Nullable. If TRUE, find only active journals.
	 * If FALSE, find only inactive journals.  If NULL, find both 
	 * active and inactive journals.
	 * 
	 * @return Never returns null.  If no matching records are found
	 * an empty set is returned.
	 */
	public List<Journal> findByAccount(long accountId, Boolean isActive) 
	{
		log.info("Enter: findByAccount("+accountId+","+isActive+")");
		
		Long min = 0L;
		
		log.info("Greater than constraint");
		Constraint<Long> constraint
			= new GreaterThan<Long>(min); 
		
		log.info("Constraint check");
		Check check
			= constraint.check(accountId);
		
		if (check != null) {
			if (check.equals(Check.LessThan)) {				
				DomainException e 
					= new DomainException(DomainExceptionCode.DOMAIN00017, accountId, min);
				log.info(e.toString());
				throw e;
			}			
			else				
			if (check.equals(Check.EqualToMin)) {
				DomainException e
					= new DomainException(DomainExceptionCode.DOMAIN00017, accountId, min);
				log.info(e.toString());
				throw e;
			}
		}		
		
		log.info("Create SQL");
		StringBuilder sp 
			= new StringBuilder();
		sp.append(" select journal from Journal journal where ");
		// account 
		sp.append(" account.id = ?1 ");
		// active
		if (isActive != null) {
			sp.append(" and isActive = ?2 " );
		}
		log.info("SQL: " + sp.toString());
		
		log.info("Create query");
		Query query 
			= getEntityManager().createQuery(sp.toString());
		
		// account
		log.info("Set parameter: " + accountId);
		query.setParameter(1, accountId);
		
		// active
		if (isActive != null) {
			log.info("Set parameter: " + isActive);
			query.setParameter(2, isActive);
		}
		
		List<Journal> dbData = null;
		
		try {
			log.info("Run query");
			dbData = (List<Journal>)query.getResultList();
		} catch (NoResultException e) {
			log.info("Query returned no results");
			dbData = new ArrayList<Journal>(0);
		}
		
		log.info("Return size: " + dbData.size());
		return dbData;
	}


	
	/**
	 * Find delete a journal.
	 * 
	 * @param accountId - required, can not be null
	 * @param journalId - required, can not be null.
	 * @return Returns the Journal object found or null
	 * if data is not found.
	 * 
	 * @throws DomainException
	 */
	public Boolean delete(long accountId, long journalId) 
	throws DomainException
	{
		log.info("Enter: delete("+accountId+","+journalId+")");
			
		log.info("Greater than constraint");
		Long min = 0L;
		Constraint<Long> constraint
			= new GreaterThan<Long>(min); 
		
		// accountId
		{
			long x = accountId;
			
			log.info("Constraint check (accountId)");
			Check check
				= constraint.check(x);
			
			if (check != null) {
				if (check.equals(Check.LessThan)) {
					DomainException e = new DomainException(DomainExceptionCode.DOMAIN00023, x, min);
					log.info(e.toString());
					throw e;
				}			
				else 
				if (check.equals(Check.EqualToMin)) {
					DomainException e = new DomainException(DomainExceptionCode.DOMAIN00023, x, min);
					log.info(e.toString());
					throw e;
				}
			}
		}
		
		// id
		{
			long x = journalId;
			
			log.info("Constraint check (journalId)");
			Check check
				= constraint.check(x);
			
			if (check != null) {
				if (check.equals(Check.LessThan)) {
					DomainException e = new DomainException(DomainExceptionCode.DOMAIN00024, x, min);
					log.info(e.toString());
					throw e;
				}			
				else 
				if (check.equals(Check.EqualToMin)) {
					DomainException e = new DomainException(DomainExceptionCode.DOMAIN00024, x, min);
					log.info(e.toString());
					throw e;
				}
			}
		}
		
		
		Journal toDelete
			= findById(accountId, journalId);
		
		if (toDelete == null) {
			log.info("The entity manager did not find accountId=\""+accountId+"\" journalId=\""+journalId+"\" so it cannot be deleted.");
			return false;
		}
		else {
			log.info("Deleting: " + toDelete);
			getEntityManager().remove(toDelete); // Should cascade and delete all children tables
			return true;
		}
	}
}
