package org.ferris.journal.ejb.journal;

import java.security.Principal;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import javax.annotation.Resource;
import javax.ejb.Local;
import javax.ejb.SessionContext;
import javax.ejb.Stateless;
import javax.persistence.EntityManager;
import javax.persistence.NoResultException;
import javax.persistence.PersistenceContext;
import javax.persistence.Query;

import org.apache.log4j.Logger;
import org.ferris.constraint.Check;
import org.ferris.constraint.Constraint;
import org.ferris.constraint.GreaterThan;
import org.ferris.constraint.NotNull;
import org.ferris.journal.ejb.lang.DomainException;
import org.ferris.journal.ejb.lang.DomainExceptionCode;
import org.ferris.sql.Like;

@Stateless(name="JournalEntryLocal")
@Local(JournalEntryLocal.class)
public class JournalEntryEjb implements JournalEntryLocal
{
	private Logger log = Logger.getLogger(getClass());
	
	@Resource
	private SessionContext sctx;
	
	/**
	 * Return EntityManager
	 * @return
	 */
	private EntityManager getEntityManager() {
		log.info("Enter: getEntityManager()");
		return entityManager;
	}
	
	@PersistenceContext(unitName="JournalPersistenceUnit")
	private EntityManager entityManager;
	
	
	/** 
	 * Default no-arg constructor
	 */
	public JournalEntryEjb() {}
	
	
	/**
	 * Set EntityManager
	 */
	public JournalEntryEjb(EntityManager entityManager) {
		this.entityManager = entityManager;
	}
	
	
	/**
	 * Find a journalentry by it's unique identifer.
	 * 
	 * @param accountId - required, can not be null
	 * @param id - required, can not be null.
	 * @return Returns the JournalEntry object found or null
	 * if data is not found.
	 * 
	 * @throws DomainException
	 */
	public JournalEntry findById(long accountId, long id) 
	throws DomainException
	{
		log.info("Enter: findId("+accountId+","+id+")");
			
		log.info("Greater than constraint");
		Long min = 0L;
		Constraint<Long> constraint
			= new GreaterThan<Long>(min); 
		
		// accountId
		{
			long x = accountId;
			
			log.info("Constraint check (accountId)");
			Check check
				= constraint.check(x);
			
			if (check != null) {
				if (check.equals(Check.LessThan)) {
					DomainException e = new DomainException(DomainExceptionCode.DOMAIN00036, x, min);
					log.info(e.toString());
					throw e;
				}			
				else 
				if (check.equals(Check.EqualToMin)) {
					DomainException e = new DomainException(DomainExceptionCode.DOMAIN00036, x, min);
					log.info(e.toString());
					throw e;
				}
			}
		}
		
		// id
		{
			long x = id;
			
			log.info("Constraint check (id)");
			Check check
				= constraint.check(x);
			
			if (check != null) {
				if (check.equals(Check.LessThan)) {
					DomainException e = new DomainException(DomainExceptionCode.DOMAIN00037, x, min);
					log.info(e.toString());
					throw e;
				}			
				else 
				if (check.equals(Check.EqualToMin)) {
					DomainException e = new DomainException(DomainExceptionCode.DOMAIN00037, x, min);
					log.info(e.toString());
					throw e;
				}
			}
		}
		
		log.info("Create query");
		Query query 
			= getEntityManager().createQuery(" select journalEntry from JournalEntry journalEntry where account.id=?1 and id=?2 ");
		
		// account id
		log.info("Set parameter: " + accountId);
		query.setParameter(1, accountId);
		
		// id
		log.info("Set parameter: " + id);
		query.setParameter(2, id);
		
		try {
			log.info("Run query");
			return (JournalEntry)query.getSingleResult();
		} catch (NoResultException e) {
			return null;
		}
	}


	/**
	 * Find delete a journal.
	 * 
	 * @param accountId - required, can not be null
	 * @param journalId - required, can not be null.
	 * @return Returns the Journal object found or null
	 * if data is not found.
	 * 
	 * @throws DomainException
	 */
	public Boolean delete(long accountId, long journalEntryId) 
	throws DomainException
	{
		log.info("Enter: delete("+accountId+","+journalEntryId+")");
			
		log.info("Greater than constraint");
		Long min = 0L;
		Constraint<Long> constraint
			= new GreaterThan<Long>(min); 
		
		// accountId
		{
			long x = accountId;
			
			log.info("Constraint check (accountId)");
			Check check
				= constraint.check(x);
			
			if (check != null) {
				if (check.equals(Check.LessThan)) {
					DomainException e = new DomainException(DomainExceptionCode.DOMAIN00040, x, min);
					log.info(e.toString());
					throw e;
				}			
				else 
				if (check.equals(Check.EqualToMin)) {
					DomainException e = new DomainException(DomainExceptionCode.DOMAIN00040, x, min);
					log.info(e.toString());
					throw e;
				}
			}
		}
		
		// id
		{
			long x = journalEntryId;
			
			log.info("Constraint check (journalEntryId)");
			Check check
				= constraint.check(x);
			
			if (check != null) {
				if (check.equals(Check.LessThan)) {
					DomainException e = new DomainException(DomainExceptionCode.DOMAIN00041, x, min);
					log.info(e.toString());
					throw e;
				}			
				else 
				if (check.equals(Check.EqualToMin)) {
					DomainException e = new DomainException(DomainExceptionCode.DOMAIN00041, x, min);
					log.info(e.toString());
					throw e;
				}
			}
		}
		
		
		JournalEntry toDelete
			= findById(accountId, journalEntryId);
		
		if (toDelete == null) {
			log.info("The entity manager did not find JournalEntry accountId=\""+accountId+"\" journalId=\""+journalEntryId+"\" so it cannot be deleted.");
			return false;
		}
		else {
			log.info("Deleting: " + toDelete);
			getEntityManager().remove(toDelete); // Should cascade and delete all children tables
			return true;
		}
	}

	
	/**
	 * Find all the entries associated with a journal
	 */
	public List<JournalEntry> findByJournal(long accountId, long journalId) throws DomainException 
	{
		log.info("Enter: findByJournal("+accountId+","+journalId+")");
		
		log.info("Greater than constraint");
		Long min = 0L;
		Constraint<Long> constraint
			= new GreaterThan<Long>(min); 
		
		// accountId
		{
			long x = accountId;
			
			log.info("Constraint check (accountId)");
			Check check
				= constraint.check(x);
			
			if (check != null) {
				if (check.equals(Check.LessThan)) {
					DomainException e = new DomainException(DomainExceptionCode.DOMAIN00042, x, min);
					log.info(e.toString());
					throw e;
				}			
				else 
				if (check.equals(Check.EqualToMin)) {
					DomainException e = new DomainException(DomainExceptionCode.DOMAIN00042, x, min);
					log.info(e.toString());
					throw e;
				}
			}
		}
		
		// journalId
		{
			long x = journalId;
			
			log.info("Constraint check (id)");
			Check check
				= constraint.check(x);
			
			if (check != null) {
				if (check.equals(Check.LessThan)) {
					DomainException e = new DomainException(DomainExceptionCode.DOMAIN00043, x, min);
					log.info(e.toString());
					throw e;
				}			
				else 
				if (check.equals(Check.EqualToMin)) {
					DomainException e = new DomainException(DomainExceptionCode.DOMAIN00043, x, min);
					log.info(e.toString());
					throw e;
				}
			}
		}
		
		log.info("Create query");
		Query query 
			= getEntityManager().createQuery(" select journalEntry from JournalEntry journalEntry where account.id=?1 and journal.id=?2 ");
		
		// account id
		log.info("Set parameter: " + accountId);
		query.setParameter(1, accountId);
		
		// journalId
		log.info("Set parameter: " + journalId);
		query.setParameter(2, journalId);
		
		List<JournalEntry> dbData = null;
		
		try {
			log.info("Run query");
			dbData = (List<JournalEntry>)query.getResultList();
		} catch (NoResultException e) {
			log.info("Query returned no results");
			dbData = new ArrayList<JournalEntry>(0);
		}
		
		log.info("Return size: " + dbData.size());
		return dbData;
	}


	/**
	 * Insert new journal entry
	 */
	public JournalEntry insert(JournalEntry journalEntry) throws DomainException 
	{
		log.info("Enter: insert(JournalEntry) " + journalEntry);
		
		log.info("Not null constraint");
		Constraint<JournalEntry> constraint
			= new NotNull<JournalEntry>(); 
		
		log.info("Contraint check");
		Check check
			= constraint.check(journalEntry);
		
		if (check != null) {			
			if (check.equals(Check.IsNull)) {
				DomainException e = new DomainException(DomainExceptionCode.DOMAIN00035);
				log.info(e.toString());
				throw e;
			}			
		}
				
		EntityManager em = getEntityManager();
		
		log.info("Persist");		
		em.persist(journalEntry);
		
		return findById(journalEntry.getAccount().getId(), journalEntry.getId());
	}


	/**
	 * Search journal entries matching the keyword.
	 */
	public List<JournalEntry> search(long accountId, String keyword,  List<Long> journalIds, Date minInclusive, Date maxInclusive)
	throws DomainException 
	{
		log.debug("Enter: search("+accountId+","+keyword+")");
		
		Principal prince = sctx.getCallerPrincipal();
		if (prince == null) {
			log.info("&&&&&&&&&&&&&& prince null!");
		} else {
			log.info("%%%%%%%%%%%%%% prince: \"" + prince.getName() +"\"");
		}
		
		EntityManager em
			= getEntityManager();
		
		int p = 0;
		
		StringBuilder sp = new StringBuilder();
		sp.append(" select journalEntry from JournalEntry journalEntry ");
		sp.append(" where ");
		
		//
		// ACCOUNT
		//
		sp.append(" account.id = ?" + (++p));
		
		//
		// MIN DAY
		//
		if (minInclusive != null) {
			sp.append(" and ");
			sp.append(" day >= ?" + (++p));
		}
		
		//
		// MAX DAY
		//
		if (maxInclusive != null) {
			sp.append(" and ");
			sp.append(" day <= ?" + (++p));
		}
		
		//
		// JOURNALS
		//
		if (journalIds != null && journalIds.size() > 0) 
		{
			sp.append(" and ");
			sp.append(" ( ");
			int i=0;
			for (Long id : journalIds) {
				if (i++ > 0) {
					sp.append(" or ");
				}
				sp.append(" journal.id = ?" + (++p));
			}
			sp.append(" ) ");
		}
		
		//
		// KEYWORD
		//
		if (keyword != null && keyword.length() > 0) 
		{
			sp.append(" and ");
			sp.append(" ( ");
			sp.append(" 	upper(subject) like ?" + (++p));
			sp.append(" 	or ");
			sp.append(" 	upper(entry) like ?" + (++p));
			sp.append(" ) ");
		}

		
		Query query 
			= em.createQuery(sp.toString());
		p=0;
		
		// 
		// ACCOUNT
		//
		query.setParameter(++p, accountId);		
		
		//
		// MIN DAY
		//
		if (minInclusive != null) {
			query.setParameter(++p, minInclusive);
		}
		
		//
		// MAX DAY
		//
		if (maxInclusive != null) {
			query.setParameter(++p, maxInclusive);
		}
		
		//
		// JOURNALS
		//
		if (journalIds != null && journalIds.size() > 0)  {
			for (Long id : journalIds) {
				query.setParameter(++p, id);
			}			
		}
		
		// 
		// KEYWORD
		//
		if (keyword != null && keyword.length() > 0) {
			String param
				= "%" + new Like(keyword).getEscapedString().toUpperCase() + "%";
			query.setParameter(++p, param);
			query.setParameter(++p, param);
		}
		
		
		List<JournalEntry> dbData = null;
		
		try {
			log.info("Run query");
			dbData = (List<JournalEntry>)query.getResultList();
		} catch (NoResultException e) {
			log.info("Query returned no results");
			dbData = new ArrayList<JournalEntry>(0);
		}
		
		log.info("Return size: " + dbData.size());
		return dbData;
	}


	/**
	 * Update existing entry
	 */
	public JournalEntry update(JournalEntry journalEntry) throws DomainException 
	{
		log.info("Enter: update(JournalEntry) " + journalEntry);
		
		log.info("Not null constraint");
		Constraint<JournalEntry> constraint
			= new NotNull<JournalEntry>(); 
		
		log.info("Constraint check");
		Check check
			= constraint.check(journalEntry);
		
		if (check != null) {
			if (check.equals(Check.IsNull)) {
				DomainException e = new DomainException(DomainExceptionCode.DOMAIN00038);
				log.info(e.toString());
				throw e;
			}			
		}
		
		EntityManager em = getEntityManager();
		
		log.info("Merge");
		em.merge(journalEntry);
		
		journalEntry = findById(journalEntry.getAccount().getId(), journalEntry.getId());
				
		log.info("Constraint check");
		check
			= constraint.check(journalEntry);
		
		if (check != null) {
			if (check.equals(Check.IsNull)) {
				DomainException e = new DomainException(DomainExceptionCode.DOMAIN00039);
				log.info(e.toString());
				throw e;
			}			
		}
		
		return journalEntry;
	}
}
