/*
 * Copyright (C) 2023 Flmelody.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.flmelody.core;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import org.flmelody.core.context.EnhancedWindwardContext;
import org.flmelody.core.context.SimpleWindwardContext;
import org.flmelody.core.exception.PluginMissException;
import org.flmelody.core.exception.ServerException;
import org.flmelody.core.netty.NettyHttpServer;
import org.flmelody.core.plugin.Plugin;
import org.flmelody.core.plugin.json.AutoJsonBinder;
import org.flmelody.core.plugin.json.JsonPlugin;
import org.flmelody.core.plugin.resolver.CompositePluginResolver;
import org.flmelody.core.plugin.resolver.PluginResolver;
import org.flmelody.core.plugin.resource.BaseStaticResourcePlugin;
import org.flmelody.core.plugin.resource.ResourcePlugin;
import org.flmelody.core.plugin.view.ViewEngineDetector;
import org.flmelody.core.plugin.view.freemarker.FreemarkerView;
import org.flmelody.core.plugin.view.groovy.GroovyView;
import org.flmelody.core.plugin.view.thymeleaf.ThymeleafView;
import org.flmelody.core.ws.WebSocketWindwardContext;
import org.flmelody.util.UrlUtil;

/**
 * @author esotericman
 */
public class Windward implements Router<Windward> {
  // Registered function
  private static final List<AbstractRouterGroup<Windward>> routerGroups = new ArrayList<>();
  // Registered resource router
  private static final List<AbstractRouterGroup<Windward>> resourceRouterGroups = new ArrayList<>();
  // Filters
  private static final List<Filter> globalFilters = new ArrayList<>();
  // Handlers for exception
  private static final List<ExceptionHandler> globalExceptionHandlers = new ArrayList<>();
  // Plugins
  private static final Map<Class<?>, Plugin> globalPlugins = new HashMap<>();
  // Root context of application
  private final String contextPath;
  // Template files location
  private final String templateRoot;
  // Static files locations
  private final String[] staticResourceLocations;
  private final PluginResolver pluginResolver = new CompositePluginResolver();
  private HttpServer httpServer;

  private Windward(String contextPath, String templateRoot, String[] staticResourceLocations) {
    this.contextPath = contextPath;
    this.templateRoot = templateRoot;
    this.staticResourceLocations = staticResourceLocations;
  }

  public static Windward setup() {
    return setup(8080, new LoggingFilter());
  }

  /**
   * Prepare core engine of Windward
   *
   * @param port server port
   * @param filters request filters
   * @return core engine of Windward
   */
  public static Windward setup(int port, Filter... filters) {
    return setup(port, UrlUtil.SLASH, filters);
  }

  /**
   * Prepare core engine of Windward
   *
   * @param port server port
   * @param contextPath path of root
   * @param filters request filters
   * @return core engine of Windward
   */
  public static Windward setup(int port, String contextPath, Filter... filters) {
    return setup(port, contextPath, "/templates", new String[] {"/static"}, filters);
  }

  /**
   * Prepare core engine of Windward
   *
   * @param port server port
   * @param contextPath path of root
   * @param templateRoot root for template files
   * @param staticResourceLocations locations of static resource
   * @param filters request filters
   * @return core engine of Windward
   */
  public static Windward setup(
      int port,
      String contextPath,
      String templateRoot,
      String[] staticResourceLocations,
      Filter... filters) {
    Windward windward = new Windward(contextPath, templateRoot, staticResourceLocations);
    windward.httpServer = new NettyHttpServer(port);
    windward
        .registerExceptionHandler(new DefaultNotFoundHandler())
        .registerFilter(filters)
        .registerPlugin(JsonPlugin.class, AutoJsonBinder.jsonPlugin)
        .registerPlugin(
            ResourcePlugin.class, new BaseStaticResourcePlugin(staticResourceLocations));
    return prepareDefault(windward);
  }

  // Prepare template engine
  private static Windward prepareDefault(Windward windward) {
    if (ViewEngineDetector.AVAILABLE_GROOVY_ENGINE) {
      windward.registerPlugin(GroovyView.class, new GroovyView());
    }
    if (ViewEngineDetector.AVAILABLE_THYMELEAF_ENGINE) {
      windward.registerPlugin(ThymeleafView.class, new ThymeleafView());
    }
    if (ViewEngineDetector.AVAILABLE_FREEMARKER_ENGINE) {
      windward.registerPlugin(FreemarkerView.class, new FreemarkerView());
    }
    return windward;
  }

  /**
   * Run server
   *
   * @throws ServerException exception
   */
  public void run() throws ServerException {
    httpServer.run();
  }

  /**
   * Define new router with specific relativePath
   *
   * @param relativePath relativePath
   * @return routerGroup
   */
  public RouterGroup<Windward> group(String relativePath) {
    DefaultRouterGroup defaultRouterGroup =
        new DefaultRouterGroup(this, UrlUtil.buildUrl(contextPath, relativePath));
    routerGroups.add(defaultRouterGroup);
    return defaultRouterGroup;
  }

  /**
   * New resource group
   *
   * @param relativePath root path
   * @return routerGroup
   */
  @SuppressWarnings("SameParameterValue")
  private RouterGroup<Windward> resourceGroup(String relativePath) {
    DefaultRouterGroup defaultRouterGroup =
        new DefaultRouterGroup(this, UrlUtil.buildUrl(contextPath, relativePath), true);
    resourceRouterGroups.add(defaultRouterGroup);
    return defaultRouterGroup;
  }

  /**
   * Register filter
   *
   * @param filters filter
   * @return current windward
   */
  public Windward registerFilter(Filter... filters) {
    if (filters == null || filters.length == 0) {
      return this;
    }
    globalFilters.addAll(Arrays.asList(filters));
    return this;
  }

  /**
   * Register exception handler
   *
   * @param exceptionHandlers exception handler
   * @return current windward
   */
  public Windward registerExceptionHandler(ExceptionHandler... exceptionHandlers) {
    if (exceptionHandlers == null || exceptionHandlers.length == 0) {
      return this;
    }
    globalExceptionHandlers.addAll(Arrays.asList(exceptionHandlers));
    globalExceptionHandlers.sort(Comparator.comparingInt(Order::getOrder));
    return this;
  }

  /**
   * Register plugin or overwrite existed
   *
   * @param clazz plugin class
   * @param plugin plugin
   * @return current windward
   */
  public Windward registerPlugin(Class<? extends Plugin> clazz, Plugin plugin) {
    // resolve plugin
    pluginResolver.resolve(this, plugin);
    // bind plugin
    globalPlugins.put(clazz, plugin);
    return this;
  }

  /**
   * Find out registered function or resource by specific path
   *
   * @param relativePath relativePath
   * @param method http methods name
   * @param <I> routers metadata
   * @return registered function
   */
  public static <I> FunctionMetaInfo<I> findRouter(String relativePath, String method) {
    for (AbstractRouterGroup<Windward> routerGroup : routerGroups) {
      FunctionMetaInfo<I> functionMetaInfo = routerGroup.matchRouter(relativePath, method);
      if (functionMetaInfo != null) {
        return functionMetaInfo;
      }
    }
    for (AbstractRouterGroup<Windward> routerGroup : resourceRouterGroups) {
      FunctionMetaInfo<I> functionMetaInfo = routerGroup.matchRouter(relativePath, method);
      if (functionMetaInfo != null) {
        return functionMetaInfo;
      }
    }
    return null;
  }

  /**
   * Get filters
   *
   * @return filters
   */
  public static List<Filter> filters() {
    return globalFilters;
  }

  /**
   * Get exception handlers
   *
   * @return exception handlers
   */
  public static List<ExceptionHandler> exceptionHandlers() {
    return globalExceptionHandlers;
  }

  /**
   * Get plugin
   *
   * @param clazz class of plugin
   * @param <T> plugin type
   * @return plugin
   */
  public static <T extends Plugin> T plugin(Class<T> clazz) {
    if (!globalPlugins.containsKey(clazz)) {
      throw new PluginMissException(String.format("Plugin [%s] not found", clazz.getName()));
    }
    //noinspection unchecked
    return (T) globalPlugins.get(clazz);
  }

  /**
   * Get plugins by super or self
   *
   * @param clazz super or self class
   * @param <T> class type
   * @return plugins
   */
  public static <T extends Plugin> List<T> plugins(Class<T> clazz) {
    return globalPlugins.values().stream()
        .filter(plugin -> clazz.isAssignableFrom(plugin.getClass()))
        .map(
            plugin -> {
              //noinspection unchecked
              return (T) plugin;
            })
        .collect(Collectors.toList());
  }

  public String getContextPath() {
    return contextPath;
  }

  public String getTemplateRoot() {
    return templateRoot;
  }

  public String[] getStaticResourceLocations() {
    return staticResourceLocations;
  }

  public Windward then() {
    return this;
  }

  /** {@inheritDoc} */
  @Override
  public <R> Windward http(HttpMethod httpMethod, String relativePath, Supplier<R> supplier) {
    group(UrlUtil.SLASH).http(httpMethod, relativePath, supplier);
    return this;
  }

  /** {@inheritDoc} */
  @Override
  public Windward http(
      HttpMethod httpMethod, String relativePath, Consumer<SimpleWindwardContext> consumer) {
    group(UrlUtil.SLASH).http(httpMethod, relativePath, consumer);
    return this;
  }

  /** {@inheritDoc} */
  @Override
  public Windward http(
      HttpMethod httpMethod, String relativePath, Function<EnhancedWindwardContext, ?> function) {
    group(UrlUtil.SLASH).http(httpMethod, relativePath, function);
    return this;
  }

  /** {@inheritDoc} */
  public <R> Windward get(String relativePath, Supplier<R> supplier) {
    group(UrlUtil.SLASH).get(relativePath, supplier);
    return this;
  }

  /** {@inheritDoc} */
  public Windward get(String relativePath, Consumer<SimpleWindwardContext> consumer) {
    group(UrlUtil.SLASH).get(relativePath, consumer);
    return this;
  }

  /** {@inheritDoc} */
  @Override
  public Windward get(String relativePath, Function<EnhancedWindwardContext, ?> function) {
    group(UrlUtil.SLASH).get(relativePath, function);
    return this;
  }

  /** {@inheritDoc} */
  @Override
  public <R> Windward put(String relativePath, Supplier<R> supplier) {
    group(UrlUtil.SLASH).put(relativePath, supplier);
    return this;
  }

  /** {@inheritDoc} */
  public Windward put(String relativePath, Consumer<SimpleWindwardContext> consumer) {
    group(UrlUtil.SLASH).put(relativePath, consumer);
    return this;
  }

  /** {@inheritDoc} */
  @Override
  public Windward put(String relativePath, Function<EnhancedWindwardContext, ?> function) {
    group(UrlUtil.SLASH).put(relativePath, function);
    return this;
  }

  /** {@inheritDoc} */
  @Override
  public <R> Windward post(String relativePath, Supplier<R> supplier) {
    group(UrlUtil.SLASH).post(relativePath, supplier);
    return this;
  }

  /** {@inheritDoc} */
  public Windward post(String relativePath, Consumer<SimpleWindwardContext> consumer) {
    group(UrlUtil.SLASH).post(relativePath, consumer);
    return this;
  }

  /** {@inheritDoc} */
  @Override
  public Windward post(String relativePath, Function<EnhancedWindwardContext, ?> function) {
    group(UrlUtil.SLASH).post(relativePath, function);
    return this;
  }

  /** {@inheritDoc} */
  @Override
  public <R> Windward delete(String relativePath, Supplier<R> supplier) {
    group(UrlUtil.SLASH).delete(relativePath, supplier);
    return this;
  }

  /** {@inheritDoc} */
  public Windward delete(String relativePath, Consumer<SimpleWindwardContext> consumer) {
    group(UrlUtil.SLASH).delete(relativePath, consumer);
    return this;
  }

  /** {@inheritDoc} */
  @Override
  public Windward delete(String relativePath, Function<EnhancedWindwardContext, ?> function) {
    group(UrlUtil.SLASH).delete(relativePath, function);
    return this;
  }

  /** {@inheritDoc} */
  @Override
  public Windward ws(String relativePath, Consumer<WebSocketWindwardContext> consumer) {
    group(UrlUtil.SLASH).ws(relativePath, consumer);
    return this;
  }

  /** {@inheritDoc} */
  @Override
  public Windward resource(String... pattern) {
    resourceGroup(UrlUtil.SLASH).resource(pattern);
    return this;
  }
}
