/*
 * Copyright (c) 2023 ForgeRock. All rights reserved.
 *
 *  This software may be modified and distributed under the terms
 *  of the MIT license. See the LICENSE file for details.
 */

package org.forgerock.android.auth.webauthn

import android.content.Context
import org.forgerock.android.auth.WebAuthnDataRepository

/**
 * Manage [PublicKeyCredentialSource] that generated by the SDK.
 * The [PublicKeyCredentialSource] only contains the reference to the actual key,
 * deleting the [PublicKeyCredentialSource] does not delete the actual key
 */
class FRWebAuthn(private val context: Context,
                 private val webAuthnDataRepository: WebAuthnDataRepository =
                     WebAuthnDataRepository.WebAuthnDataRepositoryBuilder().context(context)
                         .build()) {

    /**
     * Delete the [PublicKeyCredentialSource] by Relying Party Id
     * @param rpId The relying party id to lookup from the internal storage
     */
    fun deleteCredentials(rpId: String) {
        webAuthnDataRepository.delete(rpId)
    }

    /**
     * Delete the provide [PublicKeyCredentialSource], the [PublicKeyCredentialSource.id] will
     * be used as the key to lookup from internal storage
     * @param publicKeyCredentialSource The [PublicKeyCredentialSource] to be deleted
     */
    fun deleteCredentials(publicKeyCredentialSource: PublicKeyCredentialSource) {
        webAuthnDataRepository.delete(publicKeyCredentialSource)
    }

    /**
     * Load all the [PublicKeyCredentialSource] by the provided Relying Party Id
     * @param rpId The relying party id to lookup from the internal storage
     * @return All the [PublicKeyCredentialSource] which associate with the provide relying party id
     */
    fun loadAllCredentials(rpId: String): List<PublicKeyCredentialSource> {
        return webAuthnDataRepository.getPublicKeyCredentialSource(rpId)
    }
}